/* -*- c-file-style: "java"; indent-tabs-mode: nil -*- 
 *
 * distcc -- A simple distributed compiler system
 * $Header: /data/cvs/distcc/src/arg.c,v 1.42 2002/06/13 05:14:21 mbp Exp $ 
 *
 * Copyright (C) 2002 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


                /* "I have a bone to pick, and a few to break."
                 *     -- Anonymous */


#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/stat.h>

#include "distcc.h"
#include "trace.h"
#include "io.h"
#include "util.h"


static int dcc_argv_append(char *argv[], char *toadd)
{
    int l = dcc_argv_len(argv);
    argv[l] = toadd;
    argv[l+1] = NULL;           /* just make sure */
    return 0;
}


/**
 * Return a pointer to the extension, including the dot, or NULL.
 **/
char * dcc_find_extension(char *sfile)
{
    char *dot;

    dot = rindex(sfile, '.');
    if (dot == NULL || dot[1] == '\0') {
        /* make sure there's space for one more character after the
         * dot */
        return NULL;
    }
    return dot;
}



static int dcc_set_file_extension(const char *sfile,
                                  const char *new_ext,
                                  char **ofile)
{
    char *dot, *o;

    o = strdup(sfile);
    dot = dcc_find_extension((char *) o);
    if (!dot) {
        rs_log_error("couldn't find extension in \"%s\"", o);
        return -1;
    }
    if (strlen(dot) < strlen(new_ext)) {
        rs_log_error("not enough space for new extension");
        return -1;
    }
    strcpy(dot, new_ext);
    *ofile = o;

    return 0;
}


/**
 * If you preprocessed a file with extension @p e, what would you get?
 **/
char * dcc_preproc_exten(const char *e)
{
    if (e[0] != '.')
        return NULL;
    e++;
    if (!strcmp(e, "i") || !strcmp(e, "c")) {
        return ".i";
    } else if (!strcmp(e, "c") || !strcmp(e, "cc")
               || !strcmp(e, "cpp") || !strcmp(e, "cxx")
               || !strcmp(e, "cp") || !strcmp(e, "c++")
               || !strcmp(e, "C")) {
        return ".ii";
    } else if (!strcasecmp(e, "s")) {
        return ".s";
    } else {
        return NULL;
    }        
}


int dcc_is_preprocessed(const char *sfile)
{
    const char *dot, *ext;
    dot = dcc_find_extension((char *) sfile);
    if (!dot)
        return 0;
    ext = dot+1;

    switch (ext[0]) {
    case 's':
        /* .S needs to be run through cpp; .s does not */
        return !strcmp(ext, "s");
    case 'i':
        return !strcmp(ext, "i")
            || !strcmp(ext, "ii");
    default:
        return 0;
    }
}


/**
 * Work out whether @p sfile is source based on extension
 **/
int dcc_is_source(const char *sfile)
{
    const char *dot, *ext;
    dot = dcc_find_extension((char *) sfile);
    if (!dot)
        return 0;
    ext = dot+1;

    /* you could expand this out further into a RE-like set of case
     * statements, but i'm not sure it's that important. */

    switch (ext[0]) {
    case 'i':
        return !strcmp(ext, "i")
            || !strcmp(ext, "ii");
    case 'c':
        return !strcmp(ext, "c")
            || !strcmp(ext, "cc")
            || !strcmp(ext, "cpp")
            || !strcmp(ext, "cxx")
            || !strcmp(ext, "cp")
            || !strcmp(ext, "c++");
    case 'C':
        return !strcmp(ext, "C");
    case 's':
        return !strcmp(ext, "s");
    case 'S':
        return !strcmp(ext, "S");
    default:
        return 0;
    }

    /* XXX: Also .m is Objective-C, but I don't know if that will work
     * so for the moment we always do it locally. -- mbp */
}


/**
 * Work out the default object file name the compiler would use if -o
 * was not specified.  We don't need to worry about "a.out" or .i or
 * .s files because we've already determined that -c was specified.
 *
 * However, the compiler does put the output file in the current
 * directory even if the source file is elsewhere, so we need to strip
 * off all leading directories.
 *
 * @param sfile Source filename.  Assumed to match one of the
 * recognized patterns, otherwise bad things might happen.
 **/
int dcc_ofile_from_source(const char *sfile,
                          char **ofile)
{
    char *slash;
    int l;
    
    if ((slash = strrchr(sfile, '/')))
        sfile = slash+1;
    if ((l = strlen(sfile)) < 3) {
        rs_log_error("source file %s is bogus", sfile);
        return -1;
    }

    return dcc_set_file_extension(sfile, ".o", ofile);
}


/**
 * Parse arguments, extract ones we care about.
 *
 * This is a little hard because the cc argument rules are pretty
 * complex.
 *
 * Any options occurring before the command name must be options for
 * us.  (Do we want any of these?)
 *
 * This code is called on both the client and the server, though they
 * use the results differently.  So it does not directly cause any
 * actions, but only updates fields in @p jobinfo.
 *
 * @retval 0 if it's ok to distribute this compilation.
 *
 * @retval -1 if it would be unsafe to distribute this compilation.
 **/
int dcc_scan_args(char *argv[], char **input_file, char **output_file)
{
    int seen_opt_c = 0;
    int i;
    char *a;
    char *argstr;

    argstr = dcc_argv_tostr(argv);
    rs_trace("scanning arguments: %s", argstr);
    free(argstr);

    *input_file = *output_file = NULL;

    for (i = 0; (a = argv[i]); i++) {
        if (a[0] == '-') {
            if (!strcmp(a, "-E")) {
                rs_trace("-E call for cpp must be local");
                return -1;
            } else if (!strcmp(a, "-S")) {
                rs_log_info("-S call without assembler must be local");
                return -1;
            } else if (!strcmp(a, "-fprofile-arcs")
                       || !strcmp(a, "-ftest-coverage")) {
                rs_log_info("compiler will emit profile info; must be local");
                return -1;
            } else if (!strcmp(a, "-x")) {
                /* TODO: We could also detect options like "-x
                 * cpp-output" or "-x assembler-with-cpp", because they
                 * should override language detection based on
                 * extension.  I haven't seen anyone use them yet
                 * though. */

                rs_log_info("gcc's -x handling is complex; running locally");
                return -1;
            } else if (!strcmp(a, "-c")) {
                seen_opt_c = 1;
            } else if (!strcmp(a, "-o")) {
                /* Whatever follows must be the output */
                a = argv[++i];
                goto GOT_OUTPUT;
            } else if (str_startswith("-o", a)) {
                a += 2;         /* skip "-o" */
                goto GOT_OUTPUT;
            }
        } else {
            if (dcc_is_source(a)) {
                rs_trace("found input file \"%s\"", a);
                if (*input_file) {
                    rs_log_notice("do we have two inputs?  i give up");
                    return -1;
                }
                *input_file = a;
            } else if (str_endswith(".o", a)) {
              GOT_OUTPUT:
                rs_trace("found object file \"%s\"", a);
                if (*output_file) {
                    rs_log_notice("called for link?  i give up");
                    return -1;
                }
                *output_file = a;
            }
        }
    }

    /* TODO: ccache has the heuristic of ignoring arguments that do
     * not exist when looking for the input file; that's possibly
     * worthwile. */

    if (!seen_opt_c) {
        rs_log_notice("compiler apparently called not for compile");
        return -1;
    }

    if (!*input_file) {
        rs_log_notice("no visible input file");
        return -1;
    }

    if (!*output_file) {
        /* This is a commandline like "gcc -c hello.c".  They want
         * hello.o, but they don't say so.  For example, the Ethereal
         * makefile does this. */
        /* FIXME: This doesn't handle a.out, but that doesn't matter.
         */
        char *ofile;
        if (dcc_ofile_from_source(*input_file, &ofile))
            return -1;
        rs_log_notice("no visible output file, going to add \"%s\" at end",
                      ofile);
        dcc_argv_append(argv, "-o");
        dcc_argv_append(argv, ofile);
        *output_file = ofile;
    }

    rs_log(RS_LOG_INFO|RS_LOG_NONAME, "compile from %s to %s", *input_file, *output_file);

    return 0;
}


int
dcc_argv_len (char **a)
{
  int i;

  for (i = 0; a[i]; i++)
    ;
  return i;
}

/**
 * Adds @p delta extra NULL arguments, to allow for adding more
 * arguments later.
 **/
int dcc_shallowcopy_argv(char **from, /*@-nullstate@*/ char ***out, int delta)
{
    /*@null@*/ char **b;
    int l, i;

    assert(out != NULL);
    assert(from != NULL);

    l = dcc_argv_len(from);
    b = malloc((l+1+delta) * (sizeof from[0]));
    if (b == NULL) {
        rs_log_error("failed to allocate copy of argv");
        exit(EXIT_FAILURE);
    }
    for (i = 0; i < l; i++) {
        b[i] = from[i];
    }
    b[l] = NULL;
    
    *out = b;

    return 0;
}


/**
 * Adds @p delta extra NULL arguments, to allow for adding more
 * arguments later.
 **/
int dcc_deepcopy_argv(char **from, /*@-nullstate@*/ char ***out)
{
    char **b;
    int i, l;

    l = dcc_argv_len(from);
    
    assert(out != NULL);
    assert(from != NULL);

    l = dcc_argv_len(from);
    b = malloc((l+1) * (sizeof from[0]));
    if (b == NULL) {
        rs_log_error("failed to allocate copy of argv");
        exit(EXIT_FAILURE);
    }

    for (i = 0; i < l; i++) 
        b[i] = strdup(from[i]);
    b[l] = NULL;

    *out = b;

    return 0;
}


/**
 * Used to change "-c" to "-E", so that we get preprocessed source.
 **/
int dcc_set_action_opt(char **a, char *new_c)
{
    int gotone = 0;
    
    for (; *a; a++) 
        if (!strcmp(*a, "-c")) {
            *a = strdup(new_c);
            if (*a == NULL) {
                rs_log_error("strdup failed");
                exit(EXIT_FAILURE);
            }
            gotone = 1;
            /* keep going; it's not impossible they wrote "gcc -c -c
             * -c hello.c" */
        }

    if (!gotone) {
        rs_log_error("failed to find -c");
        return -1;
    } else {
        return 0;
    }
}



/**
 * Change object file or suffix of -o to @ofname
 *
 * It's crucially important that in every case where an output file is
 * detected by dcc_scan_args(), it's also correctly identified here.
 * It might be better to make the code shared.
 **/
int dcc_set_output(char **a, char *ofname)
{
    int i;
    
    for (i = 0; a[i]; i++) 
        if (0==strcmp(a[i], "-o")  &&  a[i+1] != NULL) {
            rs_trace("changed output from \"%s\" to \"%s\"", a[i+1], ofname);
            a[i+1] = ofname;
            rs_trace("command after: %s", dcc_argv_tostr(a));
            return 0;
        }
    /* TODO: Handle -ofoo.o */

    rs_log_error("failed to find \"-o\"");
    return -1;
}


/**
 * Change input file to @ifname; called on compiler.
 *
 * @todo Unify this with dcc_scan_args
 **/
int dcc_set_input(char **a, char *ifname)
{
    int i;
    
    rs_trace("command before: %s", dcc_argv_tostr(a));

    for (i =0; a[i]; i++)
        if (dcc_is_source(a[i])) {
            rs_trace("changed input from \"%s\" to \"%s\"", a[i], ifname);
            a[i] = ifname;
            rs_trace("command after: %s", dcc_argv_tostr(a));
            return 0;
        }
    
    rs_log_error("failed to find input file");
    return -1;
}


char *dcc_argv_tostr(char **a)
{
    int l, i;
    char *s, *ss;
    
    /* calculate total length */
    for (l = 0, i = 0; a[i]; i++) {
        l += strlen(a[i]) + 3;  /* two quotes and space */
    }

    ss = s = malloc((size_t) l + 1);
    assert(s != NULL);
    for (i = 0; a[i]; i++) {
        /* kind of half-assed quoting */
        int needs_quotes = NULL != strpbrk(a[i], " \t\n\"\';");
        if (i)
            *ss++ = ' ';
        if (needs_quotes)
            *ss++ = '"';
        strcpy(ss, a[i]);
        ss = strchr(ss, '\0');
        if (needs_quotes)
            *ss++ = '"';
    }
    *ss = '\0';

    return s;
}


