/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 * $Header: /data/cvs/distcc/src/bulk.c,v 1.9 2002/06/13 05:14:21 mbp Exp $ 
 *
 * Copyright (C) 2002 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


/**
 * @file
 *
 * Bulk file transfer, used for sending .i, .o files etc.
 *
 * @todo Optionally support gzip compression
 **/ 

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/stat.h>

#include "distcc.h"
#include "trace.h"
#include "io.h"
#include "rpc.h"


/**
 * Open a file for read, and also put its size into @p fsize.
 *
 * If the file does not exist, then returns 0, but @p ifd is -1 and @p
 * fsize is zero.  If @p fsize is zero, the caller should not try to
 * read from the file.
 *
 * This strange behaviour for ENOENT is useful because if there is
 * e.g. no output file from the compiler, we don't want to abort, but
 * rather just send nothing.  The receiver has the corresponding
 * behaviour of not creating zero-length files.
 *
 * Using fstat() helps avoid a race condition -- not a security issue,
 * but possibly a failure.  Shouldn't be very likely though.
 *
 * The caller is responsible for closing @p ifd.
 **/
int dcc_open_read(const char *fname, int *ifd, off_t *fsize)
{
    struct stat buf;
    
    *ifd = open(fname, O_RDONLY);
    if (*ifd == -1) {
        int save_errno = errno;
        if (save_errno == ENOENT) {
            /* that's OK, just assume it's empty */
            *fsize = 0;
            return 0;
        } else {
            rs_log_error("failed to open %s: %s", fname, strerror(save_errno));
            return -1;
        }
    }

    if (fstat(*ifd, &buf) == -1) {
	rs_log_error("fstat %s failed: %s", fname, strerror(errno));
        close(*ifd);
        return -1;
    }

    *fsize = buf.st_size;

    return 0;
}


/**
 * Transmit from a local file to the network.
 *
 * @param ofd File descriptor for the network connection.
 * @param fname Name of the file to send.
 * @param token Token for this file, e.g. "DOTO".
 *
 * @param size_out If non-NULL, set on return to the number of bytes
 * transmitted.
 **/
int dcc_x_file(int ofd, const char *fname, const char *token,
               size_t *size_out)
{
    int ifd;
    off_t f_size; 

    if (dcc_open_read(fname, &ifd, &f_size))
        return -1;

    rs_trace("send %ld byte file %s with token %s",
             (long) f_size, fname, token);

    if (size_out)
        *size_out = (size_t) f_size;

    if (dcc_write_str(ofd, token)
        || dcc_write_int(ofd, (unsigned) f_size))
	goto failed;

    if (f_size) {
#ifdef HAVE_SENDFILE
        if (dcc_pump_sendfile(ofd, ifd, (size_t) f_size))
            goto failed;
#else
        if (dcc_pump_readwrite(ofd, ifd, (size_t) f_size))
            goto failed;
#endif
    }

    if (ifd != -1)
        close(ifd);
    return 0;

  failed:
    if (ifd != -1)
        close(ifd);
    return -1;
}


int dcc_r_file(int ifd, const char *filename, const char *token,
               size_t *size_out)
{
    int ofd;
    int ret;

    /* XXX: tridge warns that O_CREAT ought to use O_EXCL */
    ofd = open(filename, O_TRUNC|O_WRONLY|O_CREAT, 0600);
    if (ofd == -1) {
        rs_log_error("failed to create %s: %s", filename, strerror(errno));
        return -1;
    }

    ret = dcc_r_fd(ifd, ofd, token, size_out);
    close(ofd);

    if (ret != 0) {
        rs_trace("failed to receive %s, removing it", filename);
        if (unlink(filename)) {
            rs_log_error("failed to unlink %s after failed transfer: %s",
                         filename, strerror(errno));
            return -1;
        }
    }
    
    return ret;
}


int dcc_r_fd(int ifd, int ofd, const char *token, size_t *size_out)
{
    unsigned len;

    if (dcc_expect_token(ifd, token)
        || dcc_read_int(ifd, &len))
        return -1;

    rs_trace("receive %d bytes %s file", len, token);
    if (size_out)
        *size_out = (size_t) len;

    dcc_pump_readwrite(ofd, ifd, (size_t) len);

    return 0;
}
