/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 * $Header: /data/cvs/distcc/src/distcc.c,v 1.79 2002/06/13 05:14:21 mbp Exp $ 
 *
 * Copyright (C) 2002 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


			/* 4: The noise of a multitude in the
			 * mountains, like as of a great people; a
			 * tumultuous noise of the kingdoms of nations
			 * gathered together: the LORD of hosts
			 * mustereth the host of the battle.
			 *		-- Isaiah 13 */



#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "distcc.h"
#include "trace.h"
#include "io.h"
#include "rpc.h"
#include "exitcode.h"

/* for trace.c */
const char *rs_program_name = "distcc";

/**
 * @file
 *
 * Entry point for the distcc client.
 *
 * In most parts of this program, functions return -1 for failure and
 * 0 for success.
 *
 * @todo Make sure that if we fail, the .o file is removed.
 **/


static void dcc_show_usage(void)
{
    dcc_show_version("distcc");
    dcc_show_copyright();
    printf(
"Usage:\n"
"   distcc gcc [compile options] -o OBJECT -c SOURCE\n"
"   distcc --help\n"
"\n"
"Options:\n"
"   --help                     explain usage and exit\n"
"   --version                  show version and exit\n"
"\n"
"Environment variables:\n"
"   DISTCC_HOSTS=\"HOST ...\"\n"
"            list of volunteer hosts, should include localhost\n"
"   DISTCC_VERBOSE=1           give debug messages\n"
"   DISTCC_LOG=LOGFILE         send messages here, not stderr\n"
"\n"
"distcc distributes compilation jobs across volunteer machines running\n"
"distccd.  Jobs that cannot be distributed, such as linking or \n"
"preprocessing are run locally.  distcc should be used with make's -jN\n"
"option to execute in parallel on several machines.\n"
);
}


static int dcc_write_header(int fd)
{
	return dcc_write_str(fd, "DIST")
		|| dcc_write_int(fd, PROTO_VER);
}



static int dcc_x_argv(int fd, char **argv)
{
    int i;
    int argc;
    
    argc = dcc_argv_len(argv);
    
    if (dcc_write_str(fd, "ARGC") || dcc_write_int(fd, (unsigned) argc))
        return -1;
    
    for (i = 0; i < argc; i++) {
        if (dcc_write_str(fd, "ARGV")
            || dcc_write_int(fd, (unsigned) strlen(argv[i]))
            || dcc_write_str(fd, argv[i]))
            return -1;
    }

    return 0;
}


			   
/**
 * Top-level function to run remotely.
 *
 * @param cpp_pid If nonzero, the pid of the preprocessor.  Must be
 * allowed to complete before we send the input file.
 **/
static int dcc_run_remote(char **argv, 
                          char *cpp_fname, char *output_fname,
                          pid_t cpp_pid,
                          const char *buildhost,
                          int *status)
{
    int fd;

    dcc_note_execution(buildhost, argv);
    fd = dcc_cli_connect(buildhost);
    if (fd == -1)
        return -1;
    tcp_cork_sock(fd, 1);

    if (dcc_write_header(fd)
        || dcc_x_argv(fd, argv))
        return -1;

    if (cpp_pid) {
        if (dcc_wait_child(cpp_pid, status)
            || dcc_critique_status(*status, "preprocessor", "localhost"))
            return -1;
    }

    if (dcc_x_file(fd, cpp_fname, "DOTI", NULL))
        return -1;

    tcp_cork_sock(fd, 0);
    rs_trace("client finished sending request to server");

    if (dcc_r_result_header(fd)
        || dcc_r_cc_status(fd, status)
        || dcc_r_fd(fd, STDERR_FILENO, "SERR", NULL)
        || dcc_r_fd(fd, STDOUT_FILENO, "SOUT", NULL))
        return -1;

    /* Only try to retrieve the .o file if the compiler completed
     * successfully */
    if (WIFEXITED(*status) && WEXITSTATUS(*status) == 0) {
        if (dcc_r_file(fd, output_fname, "DOTO", NULL))
            return -1;
    } else {
        rs_log_warning("compiler failed (status %x); "
                       "skipping retrieval of object file",
                       *status);
    }

    return 0;
}


/**
 * If the input filename is a plain source file rather than a
 * preprocessed source file, then preprocess it to a temporary file
 * and return the name in @p cpp_fname.
 *
 * The preprocessor may still be running when we return; you have to
 * wait for @p cpp_fid to exit before the output is complete.  This
 * allows us to overlap opening the TCP socket, which probably doesn't
 * use many cycles, with running the preprocessor.
 **/
static int dcc_cpp_maybe(char **argv, char *input_fname, char **cpp_fname,
                         pid_t *cpp_pid)
{
    char **cpp_argv;

    *cpp_pid = 0;
    
    if (dcc_is_preprocessed(input_fname)) {
        /* already preprocessed, great. */
        *cpp_fname = strdup(input_fname);
        return 0;
    }

    if (dcc_deepcopy_argv(argv, &cpp_argv))
        return -1;
    *cpp_fname = dcc_make_tmpnam("cppout", ".i");

    /* TODO: Rather than sending the output to a temporary file, get
     * the preprocessor to write to a pipe so that we can start
     * sending it directly.
     *
     * TODO: Don't wait for the preprocessor to finish before trying
     * to connect.  Instead, set it going, and then start opening the
     * remote socket, because that will probably take a while. */

    return dcc_set_action_opt(cpp_argv, "-E")
        || dcc_set_output(cpp_argv, *cpp_fname)
        || dcc_spawn_child(cpp_argv, cpp_pid, NULL, NULL);
}


int dcc_try_distributed(char *argv[], int *status, const char *buildhost)
{
    char *input_fname, *output_fname, *cpp_fname;
    pid_t cpp_pid = 0;

    /* TODO: We want to eventually make sure to always fall back to
     * compiling locally if possible, so that the disruption from a
     * damaged network is minimized. */

    return dcc_scan_args(argv, &input_fname, &output_fname)
        || dcc_cpp_maybe(argv, input_fname, &cpp_fname, &cpp_pid)
        || dcc_run_remote(argv, cpp_fname, output_fname, cpp_pid, buildhost,
                          status);
        
    return 0;
}


static void dcc_set_trace_from_env(void)
{
    const char *e;
    const char *logfile;

    if ((logfile = getenv("DISTCC_LOG"))) {
        int fd;

        rs_trace_set_level(RS_LOG_INFO);

        /* XXX: tridge warns that O_CREAT ought to use O_EXCL */
        fd = open(logfile, O_WRONLY|O_APPEND|O_CREAT, 0666);
        if (fd == -1) {
            rs_log_error("failed to open logfile %s: %s",
                         logfile, strerror(errno));
        }
        rs_trace_fd = fd;
    } else {
        rs_trace_set_level(RS_LOG_WARNING);
    }

    e = getenv("DISTCC_VERBOSE");
    if (e && *e) {
        rs_trace_set_level(RS_LOG_DEBUG);
    }
}


int main(int argc, char **argv)
{
    /* TODO: I don't think that this program needs to generally
     * rewrite its arguments; however it does need to modify the .i
     * and .o objects.  Really they need to be set to something
     * determined by the daemon process. */
    int status;
    char *buildhost;
    char **newargv;

    dcc_set_trace_from_env();
    /* TODO: Add a ping command that displays information about all
     * the servers in turn. */

    if (argc <= 1 || !strcmp(argv[1], "--help")) {
        dcc_show_usage();
        exit(0);
    } else if (!strcmp(argv[1], "--version")) {
        dcc_show_version("distcc");
        exit(0);
    } else if (argv[1][0] == '-') {
        rs_log_error("unrecognized distcc option: %s", argv[1]);
        exit(EXIT_BAD_ARGUMENTS);
    }

    argv++;

    dcc_shallowcopy_argv(argv, &newargv, 2); /* allow space for "-o" "foo.o" */

    if (dcc_pick_buildhost(&buildhost) != 0
        || strcmp(buildhost, "localhost") == 0
        || dcc_try_distributed(newargv, &status, buildhost) != 0) {
        
        buildhost = "localhost";
        dcc_note_execution(buildhost, newargv);

        /* This call execs() over the top of us.  The compiler
         * inherits stdin/out/err, so it can operate directly on them
         * if it wants. */
        
        /* XXX: I think this will leak temporary files, because we may
         * have already run the preprocessor.  We can't just delete
         * them because newargv may refer to them.  It's pretty
         * harmlesss but we should check it later. */
        dcc_run_here(newargv);
    }

    dcc_cleanup_tempfiles();
    
    return dcc_critique_status(status, "compiler", buildhost)
        ? EXIT_FAILURE : EXIT_SUCCESS;
}
