/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 * $Header: /data/cvs/distcc/src/dopt.c,v 1.13 2002/06/13 05:14:21 mbp Exp $ 
 *
 * Copyright (C) 2002 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


/**
 * @file
 *
 * Parse and apply server options.
 *
 * @todo Option to send log messages to a file, or to stderr, rather
 * than to syslog.  This will be useful for testing: we can examine
 * the log and see what happened, and also it will avoid polluting the
 * system's real log file.
 **/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <popt.h>

#include "distcc.h"
#include "trace.h"
#include "io.h"
#include "opt.h"
#include "exitcode.h"

int arg_nice_inc;
int arg_nchildren = 2;          /**< Number of children running jobs
                                 * on this machine.  About ncpus+1 I
                                 * think. */
int arg_port = 4200;

const char *arg_pid_file = NULL;


const struct poptOption options[] = {
    { "concurrent", 'n', POPT_ARG_INT, &arg_nchildren, 'n', 0, 0 },
    { "nice", 'N',       POPT_ARG_INT, &arg_nice_inc,  'N', 0, 0 },
    { "pid-file", 'P',   POPT_ARG_STRING, &arg_pid_file, 0, 0, 0 },
    { "port", 'p',       POPT_ARG_INT, &arg_port,      0, 0, 0 },
    { "help", 0,         POPT_ARG_NONE, 0, '?', 0, 0 },
    { "version", 0, POPT_ARG_NONE, 0, 'V', 0, 0 },
    { 0, 0, 0, 0, 0, 0, 0 }
};


static void distccd_show_usage(void)
{
    dcc_show_version("distccd");
    dcc_show_copyright();
    printf (
"Usage:\n"
"   distccd [OPTIONS]\n"
"\n"
"Options:\n"
"   --help                     explain usage and exit\n"
"   --version                  show version and exit\n"
#if 0
"   -n, --tasks NUM            number of concurrent connections\n"
#endif
"   -p, --port PORT            TCP port to listen on\n"
"   -P, --pid-file FILE        save daemon process id to file\n"
"   -N, --nice LEVEL           lower priority, 20=most nice\n"
"\n"
"distccd runs either from inetd or as a standalone daemon to compile\n"
"files submitted by the distcc client.  distccd runs on TCP port 4200.\n"
"\n"
"distccd should only run on trusted networks.\n"
);
}


int distccd_parse_options(int argc, const char **argv)
{
    poptContext po;
    int rc, exitcode;

    po = poptGetContext("distccd", argc, argv, options, 0);

    while ((rc = poptGetNextOpt(po)) != -1) {
        switch (rc) {
        case '?':
            distccd_show_usage();
            exitcode = 0;
            goto out_exit;

        case 'n':
            if (arg_nchildren < 1 || arg_nchildren > 20000) {
                rs_log_error("--tasks argument must be >1");
                exitcode = EXIT_BAD_ARGUMENTS;
                goto out_exit;
            }
            break;

        case 'N':
            /* just do it now */
            if (nice(arg_nice_inc) == -1) {
                rs_log_warning("nice %d failed: %s", arg_nice_inc,
                               strerror(errno));
                /* continue anyhow */
            }
            break;

        case 'V':
            dcc_show_version("distccd");
            exitcode = EXIT_SUCCESS;
            goto out_exit;
            
        default:                /* bad? */
            rs_log(RS_LOG_NONAME|RS_LOG_ERR|RS_LOG_NO_PID, "%s: %s",
                   poptBadOption(po, POPT_BADOPTION_NOALIAS),
                   poptStrerror(rc));
            exitcode = EXIT_BAD_ARGUMENTS;
            goto out_exit;
        }
    }

    poptFreeContext(po);
    return 0;

    out_exit:
    poptFreeContext(po);
    exit(exitcode);
}
