/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 * $Header: /data/cvs/distcc/src/dparent.c,v 1.6 2002/06/13 05:14:21 mbp Exp $ 
 *
 * Copyright (C) 2002 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


		/* I don't know that atheists should be considered as
		 * citizens, nor should they be considered patriots.
		 * This is one nation under God.
		 * -- Vice President, George H. W. Bush, 1987 */


/**
 * @file
 *
 * Daemon parent.  Accepts connections, forks, etc.
 *
 * @todo Quite soon we need load management.  Basically when we think
 * we're "too busy" we should stop accepting connections.  This could
 * be because of the load average, or because too many jobs are
 * running, or perhaps just because of a signal from the administrator
 * of this machine.  In that case we want to do a blocking wait() to
 * find out when the current jobs are done, or perhaps a sleep() if
 * we're waiting for the load average to go back down.  However, we
 * probably ought to always keep at least one job running so that we
 * can make progress through the queue.  If you don't want any work
 * done, you should kill the daemon altogether.
 **/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <syslog.h>
#include <signal.h>
#include <fcntl.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/socket.h>

#include "exitcode.h"
#include "distcc.h"
#include "trace.h"
#include "io.h"
#include "util.h"
#include "opt.h"

static int dcc_start_child(int listen_fd, int);
static volatile int parent_exit_signal = 0;
static void dcc_parent_loop(int listen_fd) NORETURN;
static void dcc_parent_signalled(int);
static void dcc_catch_signals(void);

static int nkids = 0;


/**
 * Be a standalone server, with responsibility for sockets and forking
 * children.  Puts the daemon in the background and detaches from the
 * controlling tty.
 **/
void dcc_standalone_server(void)
{
    int listen_fd;
    
    if ((listen_fd = open_socket_in(arg_port)) == -1)
        dcc_exit(EXIT_BIND_FAILED);

    dcc_become_daemon();

    rs_log(RS_LOG_INFO|RS_LOG_NONAME, "listening on port %d", arg_port);

    dcc_catch_signals();
    dcc_parent_loop(listen_fd);		/* noreturn */
}


/**
 * Catch and remember various fatal signals, which will cause the
 * parent and all children to exit.
 **/
static void dcc_parent_signalled(int whichsig)
{
    parent_exit_signal = whichsig;
}


/**
 * Signal handler for SIGCHLD.  Does nothing, but calling into this
 * ought to make us break out of the pause().
 **/
static void dcc_child_exited(int UNUSED(whichsig))
{
}


static void dcc_catch_signals(void)
{
    struct sigaction act_catch, act_exited;

    memset(&act_catch, 0, sizeof act_catch);
    act_catch.sa_handler = dcc_parent_signalled;

    memset(&act_exited, 0, sizeof act_exited);
    act_exited.sa_handler = dcc_child_exited;
    act_exited.sa_flags = SA_NOCLDSTOP;

    if (sigaction(SIGTERM, &act_catch, NULL)
        || sigaction(SIGHUP, &act_catch, NULL)
        || sigaction(SIGCHLD, &act_exited, NULL)) {
        rs_log_error("failed to install signal handlers: %s",
                     strerror(errno));
    }
}


static void dcc_reap_kids(void)
{
    while (1) {
        int status;
        pid_t kid;

        kid = waitpid(WAIT_ANY, &status, WNOHANG);
        if (kid == 0) {
            break;
        } else if (kid == -1) {
            if (errno == ECHILD) {
                break;
            } else {
                rs_log_error("wait failed: %s", strerror(errno));
                dcc_exit(EXIT_DISTCC_FAILED);
            }
        } else {
            /* child exited */
            --nkids;
            rs_log_info("down to %d kids", nkids);
            
            if (WIFSIGNALED(status)) {
                rs_log_error("child %d exited on signal %d",
                             (int) kid, WTERMSIG(status));
            } else if (WIFEXITED(status)) {
                rs_log_notice("child %d exited: status %d",
                              (int) kid, WEXITSTATUS(status));
            }
        }
    }
}


/**
 * Main loop for the parent process.  Basically it does nothing but
 * wait around to be signalled, or for its children to die.  What a
 * life!
 **/
static void dcc_parent_loop(int listen_fd)
{
    while (parent_exit_signal == 0) {
        int acc_fd;

        rs_log_info("waiting to accept connection");
        acc_fd = accept(listen_fd, NULL, 0);
        if (acc_fd == -1 && errno == EINTR) {
            /* just go ahead and check for children */
        }  else if (acc_fd == -1) {
            rs_log_error("accept failed: %s", strerror(errno));
            dcc_exit(EXIT_CONNECT_FAILED);
        } else {
            dcc_start_child(listen_fd, acc_fd);
            close(acc_fd);          /* in parent */
            ++nkids;
            rs_log_info("up to %d children", nkids);
        }

        dcc_reap_kids();
    }

    /* Kill our process group -- not including ourselves because we're
     * still protected. */
    rs_log_notice("terminating process group on signal %d\n", parent_exit_signal);
    if (kill(0, SIGTERM)) {
        rs_log_error("failed to kill our process group: %s", strerror(errno));
    }

    /* Remove protection, then kill ourselves */
    signal(parent_exit_signal, SIG_DFL);
    if (kill(getpid(), parent_exit_signal)) {
        rs_log_error("kill failed: %s", strerror(errno));
    }

    rs_log_notice("\"you feel very ill\"");
    dcc_exit(EXIT_SUCCESS);
}


static int dcc_start_child(int listen_fd, int accepted_fd)
{
    pid_t kid;
    
    kid = fork();
    if (kid == 0) {
        close(listen_fd);
        dcc_accept_job(accepted_fd);
        dcc_exit(EXIT_SUCCESS);
        /* doesn't return */
    } else if (kid == -1) {
        rs_log_error("fork failed: %s", strerror(errno));
        return -1;
    }

    return 0;
}


/**
 * Save the pid of the current process into the pid file, if any.
 **/
static void dcc_save_pid(void)
{
    FILE *fp;
    
    if (!arg_pid_file)
        return;

    if (!(fp = fopen(arg_pid_file, "wt"))) {
        rs_log_error("failed to open pid file: %s: %s", arg_pid_file,
                     strerror(errno));
        return;
    }

    fprintf(fp, "%ld\n", (long) getpid());

    if (fclose(fp) == -1) {
        rs_log_error("failed to close pid file: %s: %s", arg_pid_file,
                     strerror(errno));
        return;
    }
}


/**
 * Become a daemon, discarding the controlling terminal.
 *
 * Borrowed from rsync.
 **/
void dcc_become_daemon(void)
{
    int i;
    
    if (fork()) {
        _exit(0);
    }

    dcc_save_pid();
    
    /* detach from the terminal */
#ifdef HAVE_SETSID
    setsid();
#else
#ifdef TIOCNOTTY
    i = open("/dev/tty", O_RDWR);
    if (i >= 0) {
        ioctl(i, (int) TIOCNOTTY, (char *)0);      
        close(i);
    }
#endif /* TIOCNOTTY */
#endif
    /* make sure that stdin, stdout an stderr don't stuff things
       up (library functions, for example) */
    for (i=0;i<3;i++) {
        close(i); 
        open("/dev/null", O_RDWR);
    }
}
