/* -*- c-file-style: "k&r"; c-basic-offset: 4; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 * $Header: /data/cvs/distcc/src/io.c,v 1.27 2002/06/13 05:14:21 mbp Exp $ 
 *
 * Copyright (C) 2002 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


                        /* "I've always wanted to use sendfile(), but
                         * never had a reason until now"
                         *              -- mbp                        */


/**
 * @todo Perhaps also add a method of copying that truncates and mmaps
 * the destination file, and then writes directly into it.
 *
 * @todo Perhaps also add a pump method that mmaps the input file, and
 * writes from there to the output file.
 **/

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <assert.h>

#include <sys/stat.h>
#ifdef HAVE_SYS_SENDFILE_H
#  include <sys/sendfile.h>
#endif /* !HAVE_SYS_SENDFILE_H */
#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>
#include <netinet/tcp.h>

#include "distcc.h"
#include "trace.h"
#include "io.h"


#ifdef HAVE_SENDFILE
/* If you don't have it, just use dcc_pump_readwrite */

/**
 * Transmit the body of a file using sendfile().
 *
 * @param ofd Output fd
 * @param ifd Input file (must allow mmap)
 **/
int dcc_pump_sendfile(int ofd, int ifd, size_t size)
{
    if (sendfile(ofd, ifd, 0, size) != (int) size) {
	rs_log_error("sendfile failed: %s", strerror(errno));
	return -1;
    }
    return 0;
}
#endif /* def HAVE_SENDFILE */


/**
 * Copy @p n bytes from @p ifd to @p ofd.
 *
 * Does not use sendfile(), so @p ifd may be a socket.
 **/
int dcc_pump_readwrite(int ofd, int ifd, size_t n)
{
    char buf[60000], *p;
    ssize_t r_in, r_out, wanted;

    while (n > 0) {
         wanted = (n > sizeof buf) ? (sizeof buf) : n;
         r_in = read(ifd, buf, (size_t) wanted);
         
         if (r_in == -1) {
            rs_log_error("failed to read %ld bytes: %s",
                         (long) wanted, strerror(errno));
            return -1;
         } else if (r_in == 0) {
              break;	/* great */
         }
         
        n -= r_in;
        p = buf;

        while (r_in > 0) {
            r_out = write(ofd, p, (size_t) r_in);
            if (r_out == -1  ||  r_out == 0) {
                rs_log_error("failed to write: %s", strerror(errno));
                return -1;
            }
            r_in -= r_out;
            p += r_out;
        }
    }

    return 0;
}


int dcc_expect_token(int ifd, const char *token)
{
	char buf[4];

        assert(strlen(token) == 4);
        
	if (dcc_r_token(ifd, buf)) {
             rs_log_error("read failed while waiting for token \"%s\"",
                          token);
             return -1;
        }

	if (memcmp(buf, token, 4)) {
             rs_log_error("mismatch on token %s", token);
             return -1;
	}

	return 0;
}



int dcc_readx(int fd,  void *buf, size_t len)
{
	ssize_t r;
	
	while (len > 0) {
		r = read(fd, buf, len);
		if (r == -1) {
			rs_log_critical("failed to read: %s", strerror(errno));
                        return -1;
		} else if (r == 0) {
			rs_log_critical("unexpected eof on fd%d", fd);
			return -1;
		} else {
			buf = &((char *) buf)[r];
			len -= r;
		}
	}

	return 0;
}


int dcc_writex(int fd, const void *buf, size_t len)
{
	ssize_t r;
	
	while (len > 0) {
		r = write(fd, buf, len);
		if (r == -1) {
			rs_fatal("failed to write: %s", strerror(errno));
                        return -1;
		} else if (r == 0) {
			rs_fatal("unexpected eof on fd%d", fd);
			return -1;
		} else {
			buf = &((char *) buf)[r];
			len -= r;
		}
	}

	return 0;
}


int dcc_write_token(int fd, const char *what)
{
	assert(strlen(what) == 4);
	return dcc_writex(fd, what, 4);
}


int dcc_write_str(int fd, const char *what)
{
	return dcc_writex(fd, what, strlen(what));
}


int dcc_r_str_alloc(int fd, char **buf)
{
     unsigned l;
     char *s;

     if (dcc_read_int(fd, &l))
          return -1;
     assert(l > 0);

/*      rs_trace("read %d byte string", l); */

     s = *buf = malloc((size_t) l + 1);
     if (!s)
          rs_log_error("malloc failed");
     if (dcc_readx(fd, s, (size_t) l))
          return -1;

     s[l] = 0;

     return 0;
}


int dcc_write_int(int fd, unsigned v)
{
	char buf[10];
	sprintf(buf, "%08x", v);
	return dcc_writex(fd, buf, 8);
}



int dcc_read_int(int fd, unsigned *v)
{
	char buf[9], *bum;
	
	if (dcc_readx(fd, buf, 8))
		return -1;
	buf[8] = 0;

	*v = (unsigned) strtoul(buf, &bum, 16);
	if (bum != &buf[8]) {
		rs_log_error("failed to parse integer %s", buf);
		return -1;
	}

	return 0;
}



/**
 * Stick a TCP cork in the socket.  It's not clear that this will help
 * performance, but it might.
 *
 * This is a no-op if we don't think this platform has corks.
 **/
int tcp_cork_sock(int fd, int corked)
{
#ifdef TCP_CORK 
    if (setsockopt(fd, SOL_TCP, TCP_CORK, &corked, sizeof corked) == -1) {
        rs_log_notice("setsockopt failed: %s", strerror(errno));
        return -1;
    }
#endif /* def TCP_CORK */
    return 0;
}
