/* -*- c-file-style: "java"; indent-tabs-mode: nil; fill-column: 78; -*-
 * 
 * distcc -- A simple distributed compiler system
 * $Header: /data/cvs/distcc/src/lock.c,v 1.9 2003/01/27 12:23:03 mbp Exp $ 
 *
 * Copyright (C) 2002, 2003 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


                /* Power is nothing without control
                 *      -- Pirelli tyre advertisment. */


/**
 * @file
 *
 * Manage lockfiles.
 *
 * distcc uses a simple disk-based lockfile system to keep track of how many
 * jobs are queued on various machines.  These locks might be used for
 * something else in the future.
 *
 * This system has the key advantage that it doesn't require a special
 * management process and copes properly with distcc processes being killed
 * off -- the kernel automatically cleans up the lock.  It's also much nicer
 * to use than System V semaphores.
 *
 * @todo Lock files ought to incorporate port.  In fact, they need to be
 * generated in a generalized way depending on the hostdef type.
 *
 * @todo Lock files ought to also incorporate IP address, so that we can use
 * multi-A records for round-robin assignment.  On the other hand I don't
 * think that's a very promising approach...
 */

#define _GNU_SOURCE

#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <time.h>

#include <sys/stat.h>
#include <sys/file.h>

#include "distcc.h"
#include "trace.h"
#include "util.h"
#include "hosts.h"
#include "tempfile.h"
#include "lock.h"
#include "io.h"


struct dcc_hostdef _dcc_local = {
    DCC_MODE_LOCAL,
    NULL,
    "localhost",
    0,
    NULL,
    4,                          /* number of tasks */
    NULL    
};

struct dcc_hostdef *dcc_hostdef_local = &_dcc_local;

static char * dcc_make_lock_filename(const char *host, int iter)
{
    int need_len;
    char * buf;
    const char *tempdir;

    tempdir = dcc_get_tempdir();
    need_len = strlen(tempdir) + 6 + strlen(host) + 1 + 7 + 1;
    buf = malloc(need_len);
    if (snprintf(buf, need_len, "%s/lock_%s_%07d", tempdir, host, iter)
        != need_len - 1) {
        rs_fatal("wrong length??");
    }
    return buf;
}


/**
 * Get an exclusive, non-blocking lock on a file using whatever method
 * is available on this system.
 *
 * @return 0 if we got the lock; -1 with EAGAIN if the file is already
 * locked.
 **/
static int sys_lock(int fd, int block)
{
#if defined(F_SETLK)
    struct flock lockparam;

    lockparam.l_type = F_WRLCK;
    lockparam.l_whence = SEEK_SET;
    lockparam.l_start = 0;
    lockparam.l_len = 0;        /* whole file */
    
    return fcntl(fd, block ? F_SETLKW : F_SETLK, &lockparam);
#elif defined(HAVE_FLOCK)
    return flock(fd, LOCK_EX | (block ? 0 : LOCK_NB));
#elif defined(HAVE_LOCKF)
    return lockf(fd, block ? F_LOCK : F_TLOCK, 0);
#else
#  error "No supported lock method.  Please port this code."
#endif
}


int dcc_lock_host(const struct dcc_hostdef *host, int slot, int block)
{
    const char *tempdir;
    char *fname;
    int fd = -1;
    int ret;

    tempdir = dcc_get_tempdir();
    fname = dcc_make_lock_filename(host->hostname, slot);

    /* Create if it doesn't exist.  We don't actually do anything with
     * the file except lock it.*/
    if ((fd = open(fname, O_WRONLY|O_CREAT, 0600)) == -1
        && errno != EEXIST) {
        rs_log_error("failed to creat %s: %s", fname, strerror(errno));
        goto bomb;
    }

    ret = sys_lock(fd, block);
    
    if (ret != -1) {
        rs_trace("locked %s", fname);
        free(fname);
        return 0;
    } else
        switch (errno) {
#ifdef EWOULDBLOCK
        case EWOULDBLOCK:
#endif
#if defined(EAGAIN) && EAGAIN != EWOULDBLOCK
        case EAGAIN:
#endif
            rs_trace("%s already locked", fname);
            break;
        default:
            rs_log_error("lock %s failed: %s", fname, strerror(errno));
            break;
    }

    bomb:
    if (fd != -1)
        dcc_close(fd);
    free(fname);
    return -1;
}
