
import time, sys, string, os, types, unittest, re, popen2, pprint
import signal, os.path, string
import comfychair

class SimpleDistCC_Case(comfychair.TestCase):
    '''Abstract base class for distcc tests'''
    def __init__(self):
        self.cmd_log = ''

    def setUp(self):
        self.enterRundir()
        self.stripEnvironment()

    def enterRundir(self):
        """Create and enter a temporary directory for this test."""
        self.basedir = os.getcwd()
        self.rundir = os.path.join(self.basedir,
                                   'testtmp', 
                                   self.__class__.__name__)
        self.tmpdir = os.path.join(self.rundir, 'tmp')
        os.system("rm -fr %s" % self.rundir)
        os.makedirs(self.tmpdir)
        os.system("mkdir -p %s" % self.rundir)
        os.chdir(self.rundir)

    def stripEnvironment(self):
        """Remove all DISTCC variables from the environment, so that
        the test is not affected by the development environment."""
        for key in os.environ.keys():
            if key[:7] == 'DISTCC_':
                # NOTE: This only works properly on Python 2.2: on
                # earlier versions, it does not call unsetenv() and so
                # subprocesses may get confused.
                del os.environ[key]
        os.environ['TMPDIR'] = self.tmpdir

    def tearDown(self):
        self.leaveRundir()

    def leaveRundir(self):
        os.chdir(self.basedir)

    def explainFailure(self, exc_info):
        import traceback
        print "-----------------------------------------------------------------"
        traceback.print_exc(file=sys.stdout)
        print self.cmd_log
        print "-----------------------------------------------------------------"
    
    def fail(self, msg):
        raise AssertionError(msg)

    def assertEquals(self, a, b):
        if not a == b:
            raise AssertionError("assertEquals failed: %s" % `(a, b)`)
            
    def assertNotEqual(self, a, b):
        if a == b:
            raise AssertionError("assertNotEqual failed: %s" % `(a, b)`)

    def assertReMatch(self, needle, haystack):
        """Look for the regular expression re in haystack"""
        if not re.search(needle, haystack):
            self.fail("couldn't find re %s in %s" % (`needle`, `haystack`))
            

class RunCmd_Case(SimpleDistCC_Case):
    '''A test case that involves running distcc and looking at the output.'''
    def __init__(self):
        SimpleDistCC_Case.__init__(self)
        self.background_pids = []

    
    def runCmdUnchecked(self, cmd, mcheck=1):
        '''Invoke a distcc command; return (exitcode, stdout)'''
        if mcheck:
            cmd = "MALLOC_CHECK_=2 " + cmd
        pobj = popen2.Popen4(cmd)
        output = pobj.fromchild.read()
        waitstatus = pobj.wait()
        assert not os.WIFSIGNALED(waitstatus), ("%s terminated with signal %d",
                                                cmd, os.WTERMSIG(waitstatus))
        rc = os.WEXITSTATUS(waitstatus)
        self.cmd_log += ("""Run command: %s
Wait status: %#x
Output:
%s""" % (cmd, waitstatus, output))
        return rc, output


    def runCmd(self, cmd, expectedResult = 0):
        rc, output = self.runCmdUnchecked(cmd)
        if rc != expectedResult:
            self.fail("command returned %d; expected %s: \"%s\"" %
                      (rc, expectedResult, cmd))
            
        return output


    def runCmdNoWait(self, cmd):
        name = cmd[0]
        self.cmd_log += "Run in background:\n" + `cmd` + "\n"
        pid = os.spawnvp(os.P_NOWAIT, name, cmd)
        self.cmd_log += "pid: %d\n" % pid
        return pid


class WithDaemon_Case(RunCmd_Case):
    """Start the daemon, and then run a command locally against it.

The daemon doesn't detach until it has bound the network interface, so
as soon as that happens we can go ahead and start the client."""

    def setUp(self):
        RunCmd_Case.setUp(self)
        self.daemon_pidfile = os.path.join(os.getcwd(), "daemonpid.tmp")
        self.daemon_logfile = os.path.join(os.getcwd(), "distccd.log")
        self.server_port = 42000
        self.startDaemon()
        self.setupEnv()

    def setupEnv(self):
        os.environ['DISTCC_HOSTS'] = '127.0.0.1:%d' % self.server_port
        os.environ['DISTCC_LOG'] = os.path.join(os.getcwd(), 'distcc.log')
        os.environ['DISTCC_VERBOSE'] = '1'

    def tearDown(self):
        self.killDaemon()
        RunCmd_Case.tearDown(self)


    def killDaemon(self):
        import signal, time
        pid = int(open(self.daemon_pidfile, 'rt').read())
        os.kill(pid, signal.SIGTERM)

        # We can't wait on it, because it detached.  So just keep
        # pinging until it goes away.
        while 1:
            try:
                os.kill(pid, 0)
            except OSError:
                break
            time.sleep(0.2)


    def startDaemon(self):
        """Start a daemon in the background, return its pid"""
        # The daemon detaches once it has successfully bound the
        # socket, so if something goes wrong at startup we ought to
        # find out straight away.  If it starts successfully, then we
        # can go ahead and try to connect.
        
        cmd = ("distccd --verbose --daemon --log-file %s --pid-file %s --port %d"
               % (self.daemon_logfile, self.daemon_pidfile, self.server_port))
        self.runCmd(cmd)



