#! /usr/bin/env python

"""comfychair: a Python-based instrument of software torture.


Copyright (C) 2002 by Martin Pool <mbp@samba.org>

This software may be used, modified and distributed without
restriction, except that this copyright notice must be retained.  This
software comes with ABSOLUTELY NO WARRANTY of any kind.


This is a test framework designed for testing programs written in
Python, or (through a fork/exec interface) any other language.  It is
similar in design to the very nice 'svntest' system used by
Subversion, but has no Subversion-specific features.

It is somewhat similar to PyUnit, except:

 - it allows capture of detailed log messages from a test, to be
   optionally displayed if the test fails.

 - it allows execution of a specified subset of tests

 - it avoids Java idioms that are not so useful in Python

WRITING TESTS:

  Each test case is a callable object, typically a function.  Its
  documentation string describes the test, and the first line of the
  docstring should be a brief name.

  The test should return 0 for pass, or non-zero for failure.
  Alternatively they may raise an exception.

  Tests may import this "comfychair" module to get some useful
  utilities, but that is not strictly required.
  
"""

# TODO: Put everything into a temporary directory?

# TODO: Have a means for tests to customize the display of their
# failure messages.  In particular, if a shell command failed, then
# give its stderr.

import sys

class TestCase:
    """A base class for tests.  This class defines required functions which
    can optionally be overridden by subclasses.  It also provides some
    utility functions for"""

    def __init__(self):
        self.cmd_log = ""
    
    def setUp(self):
        """Set up test fixture."""
        pass

    def tearDown(self):
        """Tear down test fixture."""
        pass

    def runTest(self):
        """Run the test."""
        pass

    def fail(self, reason = ""):
        """Say the test failed."""
        raise AssertionError(reason)

    def assert_re_match(self, pattern, s):
        """Assert that a string matches a particular pattern

        Inputs:
          pattern      string: regular expression
          s            string: to be matched

        Raises:
          AssertionError if not matched
          """
        import re
        if not re.match(pattern, s):
            raise AssertionError("string %s does not match regexp %s" % (`s`, `pattern`))

    def assert_regexp(self, pattern, s):
        """Assert that a string *contains* a particular pattern

        Inputs:
          pattern      string: regular expression
          s            string: to be searched

        Raises:
          AssertionError if not matched
          """
        import re
        if not re.search(pattern, s):
            raise AssertionError("string %s does not contain regexp %s" % (`s`, `pattern`))


    def assert_no_file(self, filename):
        import os.path
        assert not os.path.exists(filename), ("file exists but should not: %s" % filename)

    def runCmd(self, cmd, expectedResult = 0):
        """Run a command, fail if the command returns an unexpected exit
        code.  Return the output produced."""
        rc, output = self.runCmdUnchecked(cmd)
        if rc != expectedResult:
            raise AssertionError("command returned %d; expected %s: \"%s\"" %
                                 (rc, expectedResult, cmd))

        return output

    def runCmdUnchecked(self, cmd):
        """Invoke a command; return (exitcode, stdout)"""
        import os, popen2
        pobj = popen2.Popen4(cmd)
        output = pobj.fromchild.read()
        waitstatus = pobj.wait()
        assert not os.WIFSIGNALED(waitstatus), \
               ("%s terminated with signal %d", cmd, os.WTERMSIG(waitstatus))
        rc = os.WEXITSTATUS(waitstatus)
        self.cmd_log += ("""Run command: %s
Wait status: %#x
Output:
%s""" % (cmd, waitstatus, output))
        return rc, output

    def explainFailure(self, exc_info):
        import traceback
        print "-----------------------------------------------------------------"
        traceback.print_exc(file=sys.stdout)
        print self.cmd_log
        print "-----------------------------------------------------------------"

def runtests(test_list):
    """Run a series of tests.

    Eventually, this routine will also examine sys.argv[] to handle
    extra options.

    Inputs:
      test_list    sequence of callable test objects

    Returns:
      nothing
    """

    def test_name(test):
        """Return a human-readable name for a test.

        Inputs:
          test         some kind of callable test object

        Returns:
          name         string: a short printable name
          """
        try:
            return test.__name__
        except:
            return `test`

    import traceback
    for test in test_list:
        print "%-60s" % test_name(test),
        # flush now so that long running tests are easier to follow
        sys.stdout.flush()

        obj = test()
        
        try:
            try: # run test and show result
                if hasattr(obj, "setUp"):
                    obj.setUp()
                obj.runTest()
                print "OK"
            except KeyboardInterrupt:
                print "INTERRUPT"
                obj.explainFailure(sys.exc_info())
                break
            except AssertionError:
                print "FAIL"
                obj.explainFailure(sys.exc_info())
        finally:
            try:
                # XXX: This is not really very clean, but at least it avoids masking
                # errors that happened during testing, which are probably more
                # important.
                if hasattr(obj, "tearDown"):
                    obj.tearDown()
            except KeyboardInterrupt:
                print "interrupted during tearDown"
                obj.explainFailure(sys.exc_info())
                break

if __name__ == '__main__':
    print __doc__
