/* -*- c-file-style: "java"; indent-tabs-mode: nil; fill-column: 78; -*-
 * 
 * distcc -- A simple distributed compiler system
 *
 * Copyright (C) 2002, 2003 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


                        /* I just wish I could get caller-IQ on my phones...
                                   -- The Purple People-Eater, NANAE */



#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>
#include <signal.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>

#include <netinet/in.h>
#include <netinet/tcp.h>

#include <netdb.h>

#include "types.h"
#include "distcc.h"
#include "trace.h"
#include "io.h"
#include "exitcode.h"
#include "clinet.h"
#include "util.h"


/**
 * @file
 *
 * Client-side networking.
 *
 * @todo In error messages, show the name of the relevant host.
 * Should do this even in readx(), etc.
 *
 * @todo Connect can take a very long time to fail.  We probably need a
 * userspace timeout on it, e.g. by doing a nonblocking connect and select()
 * on it.  Check how to do this properly.
 **/



/**
 * @p host and @p port are only here to aid printing debug messages.
 **/
static int dcc_connect_by_addr(const char *host, int port,
                               struct sockaddr *sa, size_t sa_len,
                               int *p_fd)
{
    int fd;

    fd = socket(sa->sa_family, SOCK_STREAM, 0);
    
    if (fd == -1) {
	rs_log_error("failed to create socket: %s", strerror(errno));
	return EXIT_CONNECT_FAILED;
    }

    if (connect(fd, sa, sa_len)) {
        /* TODO: Print raw sockaddr form in some readable way. */
        rs_log_error("failed to connect to %s port %d: %s", host, port, 
                     strerror(errno));
        close(fd);
	return EXIT_CONNECT_FAILED;
    }

        /* TODO: Print raw sockaddr form in some readable way. */
    rs_trace("client got connection to %s port %d on fd%d", host, port, fd);

    *p_fd = fd;
    return 0;
}


#if defined(ENABLE_RFC2553)

/**
 * Open a socket to a tcp remote host with the specified port.
 *
 * @todo Don't try for too long to connect. 
 **/
int dcc_connect_by_name(const char *host, int port, int *p_fd)
{
    struct addrinfo hints;
    struct addrinfo *res;
    int error;
    int ret;
    char portname[20];

    /* Unfortunately for us, getaddrinfo wants the port (service) as a string */
    snprintf(portname, sizeof portname, "%d", port);

    memset(&hints, 0, sizeof(hints));
    /* set-up hints structure */
    hints.ai_family = PF_UNSPEC;
    hints.ai_socktype = SOCK_STREAM;
    error = getaddrinfo(host, portname, &hints, &res);
    if (error) {
        rs_log_error("failed to resolve host %s port %d: %s", host, port,
                     gai_strerror(error));
        return EXIT_CONNECT_FAILED;
    }

    /* Try each of the hosts possible addresses. */
    do {
        ret = dcc_connect_by_addr(host, port, res->ai_addr, res->ai_addrlen,
                                  p_fd);
    } while (ret != 0 && (res = res->ai_next));

    return ret;
}


#else /* not ENABLE_RFC2553 */

/**
 * Open a socket to a tcp remote host with the specified port.
 *
 * @todo Don't try for too long to connect. 
 **/
int dcc_connect_by_name(const char *host, int port, int *p_fd)
{
    struct sockaddr_in sock_out;
    struct hostent *hp;

    /* FIXME: "warning: gethostbyname() leaks memory.  Use gethostbyname_r
     * instead!" (or indeed perhaps use getaddrinfo?) */
    hp = gethostbyname(host);
    if (!hp) {
	rs_log_error("failed to look up host \"%s\": %s", host,
                     hstrerror(h_errno));
	return EXIT_CONNECT_FAILED;
    }

    memcpy(&sock_out.sin_addr, hp->h_addr, (size_t) hp->h_length);
    sock_out.sin_port = htons((in_port_t) port);
    sock_out.sin_family = PF_INET;

    return dcc_connect_by_addr(host, port, (struct sockaddr *) &sock_out,
                               sizeof sock_out, p_fd);
}

#endif /* not ENABLE_RFC2553 */
