/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 *
 * Copyright (C) 2003 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "config.h"

#include <sys/types.h>
#include <sys/stat.h>

#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>

#include <dirent.h>

#include "types.h"
#include "rpc.h"
#include "io.h"
#include "distcc.h"
#include "trace.h"
#include "exitcode.h"
#include "state.h"
#include "tempfile.h"
#include "filename.h"
#include "snprintf.h"
#include "mon.h"
#include "util.h"
#include "netutil.h"


/**
 * @file
 *
 * Common routines for monitoring compiler state.
 *
 * Every time the client wants an update, it can call dcc_mon_poll(),
 * which returns a newly allocated list of all running processes.
 **/


/* TODO: Shouldn't fail if the directory doesn't exist at the moment.
 *
 * TODO: Don't emit visible warnings for transient failures?
 */


/*
 * An mtime older than 30s would imply that no progress had been made
 * in that time, which sounds pretty unlikely to me.
 */
const int dcc_state_max_age = 30;


/**
 * Check if the state file @p fd is too old to be believed -- probably
 * because it was left over from a client that was killed.
 *
 * If so, close @p fd, unlink the file, and return EXIT_GONE.
 *
 * fd is closed on failure.
 **/
static int dcc_mon_kill_old(int fd,
                            char *fullpath)
{
    struct stat st;
    time_t now;

    /* Check if the file is old. */
    if (fstat(fd, &st) == -1) {
        dcc_close(fd);
        rs_log_warning("error statting %s: %s", fullpath, strerror(errno));
        return EXIT_IO_ERROR;
    }
    time(&now);

    /* Time you hear the siren / it's already too late */
    if (now - st.st_mtime > dcc_state_max_age) {
        dcc_close(fd);          /* close first for windoze */
        rs_trace("unlink %s", fullpath);
        if (unlink(fullpath) == -1) {
            rs_log_warning("unlink %s failed: %s", fullpath, strerror(errno));
            return EXIT_IO_ERROR;
        }
        return EXIT_GONE;
    }

    return 0;
}


static int dcc_mon_read_state(int fd, char *fullpath,
                              struct dcc_mon_list *lp)
{
    int ret;
    unsigned val;

    if ((ret = dcc_r_token_int(fd, "STFV", &val)))
        return ret;

    if (val != 0) {
        rs_log_warning("bad state file version in %s",
                       fullpath);
        return EXIT_PROTOCOL_ERROR;
    }

    if ((ret = dcc_r_token_int(fd, "CPID", &lp->cpid)))
        return ret;

    if ((ret = dcc_r_token_string(fd, "STTE", &lp->state)))
        return ret;

    if ((ret = dcc_r_token_string(fd, "FILE", &lp->file)))
        return ret;

    if ((ret = dcc_r_token_string(fd, "HOST", &lp->host)))
        return ret;

    return 0;
}


/**
 * Check that the process named by the file still exists; if not,
 * return EXIT_GONE.
 **/
static int dcc_mon_check_orphans(struct dcc_mon_list *monl)
{
    return kill(monl->cpid, 0) ? EXIT_GONE : 0;
}


/**
 * Read state.  If loaded successfully, store a pointer to the newly
 * allocated structure into *ppl.
 */
static int dcc_mon_load_state(int fd,
                              char *fullpath,
                              struct dcc_mon_list **ppl)
{
    int ret;
    struct dcc_mon_list *tl;

    tl = calloc(1, sizeof *tl);
    if (!tl)
        rs_fatal("failed to allocate dcc_mon_list");
    
    ret = dcc_mon_read_state(fd, fullpath, tl);

    if (ret == 0)
        ret = dcc_mon_check_orphans(tl);

    if (ret) {
        dcc_mon_list_free(tl);
    } else {
        *ppl = tl;
    }
    
    return ret;
}


/* Free the whole list */
int dcc_mon_list_free(struct dcc_mon_list *lp)
{
    struct dcc_mon_list *next;

    while (lp) {
        next = lp->next;        /* save from clobbering */
        
        free(lp->file);
        free(lp->host);
        free(lp->state);
        free(lp);

        lp = next;
    }
    
    return 0;
}


/**
 * Read in @p filename from inside @p dirname, and try to parse it as
 * a status file.
 *
 * If a new entry is read, a pointer to it is returned in @p lp.
 **/
static int dcc_mon_do_file(char *dirname, char *filename,
                           struct dcc_mon_list **lp)
{
    int fd;
    char *fullpath;
    int ret;

    *lp = NULL;
    
    /* Is this a file we want to see */
    if (!str_startswith(dcc_state_prefix, filename)) {
/*         rs_trace("skipped"); */
        return 0;
    }

    asprintf(&fullpath, "%s/%s", dirname, filename);
    rs_trace("process %s", fullpath);

    /* Remember that the file might disappear at any time, so open it
     * now so that we can hang on. */
    if ((fd = open(fullpath, O_RDONLY|O_BINARY, 0)) == -1) {
        if (errno == ENOENT) {
            rs_trace("%s disappeared", fullpath);
            ret = 0;
            goto out_free;
        } else { /* hm */
            rs_log_warning("failed to open %s: %s",
                           fullpath, strerror(errno));
            ret = EXIT_IO_ERROR;
            goto out_free;
        }
    }

    if ((ret = dcc_mon_kill_old(fd, fullpath))) {
        /* closes fd on failure */
        goto out_free;
    }

    ret = dcc_mon_load_state(fd, fullpath, lp);
    
    dcc_close(fd);
    
    out_free:    
    free(fullpath);
    return ret;           /* ok */
}


/**
 * Read through the state directory and return information about all
 * processes we find there.
 *
 * This function has to handle any files in there that happen to be
 * corrupt -- that can easily happen if e.g. a client crashes or is
 * interrupted, or is even just in the middle of writing its file.
 **/
int dcc_mon_poll(struct dcc_mon_list **p_list)
{
    int ret;
    char *dirname;
    DIR *d;
    struct dirent *de;
    struct dcc_mon_list **ppnext;

    *p_list = NULL;
    
    if ((ret = dcc_get_state_dir(&dirname)))
        return ret;

    if ((d = opendir(dirname)) == NULL) {
        rs_log_error("failed to opendir %s: %s", dirname, strerror(errno));
        ret = EXIT_IO_ERROR;
        return ret;
    }

    ppnext = p_list;
    while ((de = readdir(d)) != NULL) {
        struct dcc_mon_list *pthis;
        if (dcc_mon_do_file(dirname, de->d_name, &pthis) == 0
            && pthis) {
            /* We can succeed without getting a new entry back, but it
             * turns out that this time we did get one.  So store a
             * pointer to it onto the list, and prepare to store the
             * next pointer into this node's link slot. */
            *ppnext = pthis;
            ppnext = &(pthis->next);
        }
    }

    closedir(d);

    return 0;    
}


