/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 *
 * Copyright (C) 2002, 2003 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

/**
 * @file
 *
 * Send a compilation request to a remote server.
 **/


#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/types.h>

#include "distcc.h"
#include "trace.h"
#include "io.h"
#include "rpc.h"
#include "exitcode.h"
#include "util.h"
#include "clinet.h"
#include "hosts.h"
#include "tempfile.h"
#include "exec.h"
#include "lock.h"
#include "clirpc.h"
#include "timeval.h"
#include "compile.h"
#include "state.h"


/**
 * Pass a compilation across the network.
 *
 * When this function is called, the preprocessor has already been
 * started in the background.  It may have already completed, or it
 * may still be running.  The goal is that preprocessing will overlap
 * with setting up the network connection, which may take some time
 * but little CPU.
 *
 * If this function fails, compilation will be retried on the local
 * machine.
 *
 * @param argv Compiler command to run.
 *
 * @param cpp_fname Filename of preprocessed source.  May not be complete yet,
 * depending on @p cpp_pid.
 *
 * @param output_fname File that the object code should be delivered to.
 * 
 * @param cpp_pid If nonzero, the pid of the preprocessor.  Must be
 * allowed to complete before we send the input file.
 *
 * @param host Definition of host to send this job to.
 *
 * @param status on return contains the wait-status of the remote
 * compiler.
 *
 * @returns 0 on success, otherwise error.  Returning nonzero does not
 * necessarily imply the remote compiler itself succeeded, only that
 * there were no communications problems. 
 **/
int dcc_compile_remote(char **argv, 
                       char *cpp_fname,
                       char *output_fname,
                       pid_t cpp_pid,
                       struct dcc_hostdef *host,
                       int *status)
{
    int to_net_fd, from_net_fd;
    int ret;
    pid_t ssh_pid = 0;
    int ssh_status;

    /* For ssh support, we need to allow for separate fds writing to and
     * reading from the network, because our connection to the ssh client may
     * be over pipes, which are one-way connections. */

    *status = 0;

    dcc_note_state(STATE_CONNECT, NULL, host->hostdef_string);
    
    dcc_note_execution(host, argv);
    if (host->mode == DCC_MODE_TCP) {
        if ((ret = dcc_connect_by_name(host->hostname, host->port,
                                       &to_net_fd)) != 0)
            return ret;
        from_net_fd = to_net_fd;
    } else if (host->mode == DCC_MODE_SSH) {
        if ((ret = dcc_ssh_connect(NULL, host->user, host->hostname,
                                   host->ssh_command,
                                   &from_net_fd, &to_net_fd,
                                   &ssh_pid)))
            return ret;
    } else {
        rs_fatal("impossible host mode");
    }

    /* This waits for cpp and puts its status in *status.  If cpp failed, then
     * the connection will have been dropped and we need not bother trying to
     * get any response from the server. */
    ret = dcc_send_job_corked(to_net_fd, argv, cpp_pid, status, cpp_fname);

    /* TODO: If cpp finishes early and fails then perhaps break out of trying
     * to connect.  */

    /* OK, now all of the source has at least made it into the
     * client's TCP transmission queue, sometime soon the server will
     * start compiling it.
     */
    dcc_note_state(STATE_COMPILE, NULL, host->hostdef_string);

    if (to_net_fd != from_net_fd) {
        /* in ssh mode, we can start closing down early */
        dcc_close(to_net_fd);
    }

    if (ret == 0 && *status == 0) {
        ret = dcc_retrieve_results(from_net_fd, status, output_fname);
    }

    /* Close socket so that the server can terminate, rather than
     * making it wait until we've finished our work. */
    dcc_close(from_net_fd);

    /* Collect the SSH child.  Strictly this is unnecessary; it might slow the
     * client down a little when things could otherwise be proceeding in the
     * background.  But it helps make sure that we don't assume we succeeded
     * when something possibly went wrong, and it allows us to account for the
     * cost of the ssh child. */
    if (ssh_pid)
        dcc_collect_child("ssh", ssh_pid, &ssh_status); /* ignore failure */

    /* TODO: Check status from SSH; fail if it's bad.  (Would failing
     * actually improve the sum of human happiness?  If we got a
     * complete response what more can we say?) */

    return ret;
}


