/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 *
 * Copyright (C) 2002, 2003 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


			/* 4: The noise of a multitude in the
			 * mountains, like as of a great people; a
			 * tumultuous noise of the kingdoms of nations
			 * gathered together: the LORD of hosts
			 * mustereth the host of the battle.
			 *		-- Isaiah 13 */



#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include "distcc.h"
#include "trace.h"
#include "exitcode.h"
#include "util.h"
#include "hosts.h"
#include "bulk.h"
#include "tempfile.h"
#include "implicit.h"
#include "compile.h"
#include "state.h"
#include "arg.h"

/* Name of this program, for trace.c */
const char *rs_program_name = "distcc";


/**
 * @file
 *
 * Entry point for the distcc client.
 *
 * In most parts of this program, functions return 0 for success and something
 * from exitcode.h for failure.  However it is not completely consistent.
 *
 * @todo Make absolutely sure that if we fail, the .o file is removed.
 * Perhaps it would be better to receive to a temporary file and then
 * rename into place?  On the other hand, gcc seems to just write
 * directly, and if we fail or crash then Make ought to know not to
 * use it.
 *
 * @todo Count the preprocessor, and any compilations run locally, against the
 * load of localhost.  In doing this, make sure that we cannot deadlock
 * against a load limit, by having a case where we need to hold one lock and
 * take another to make progress.  I don't think there should be any such case
 * -- we can release the cpp lock before starting the main compiler.
 *
 * @todo If we have produced a .i file and need to fall back to running
 * locally then use that rather than the original source.
 **/


static void dcc_show_usage(void)
{
    dcc_show_version("distcc");
    dcc_show_copyright();
    printf(
"Usage:\n"
"   distcc [COMPILER] [compile options] -o OBJECT -c SOURCE\n"
"   distcc --help\n"
"\n"
"Options:\n"
"   COMPILER                   defaults to \"cc\"\n"
"   --help                     explain usage and exit\n"
"   --version                  show version and exit\n"
"\n"
"Environment variables:\n"
"   DISTCC_VERBOSE=1           give debug messages\n"
"   DISTCC_LOG=LOGFILE         send messages here, not stderr\n"
"   DISTCC_TCP_CORK=0          disable TCP corks\n"
"   DISTCC_SSH=cmd             command to run to open SSH connections\n"
"\n"
"Server specification:\n"
"A list of servers is taken from the environment variable $DISTCC_HOSTS, or\n"
"the file $HOME/.distcc/hosts, or the file %s/distcc/hosts.\n"
"Each host can be given in any of these forms, see the manual for details:\n"
"\n"
"   localhost                  run in place\n"
"   HOST                       TCP connection, port %d\n"
"   HOST:PORT                  TCP connection, specified port\n"
"   @HOST                      SSH connection\n"
"   USER@HOST                  SSH connection to specified host\n"
"\n"
"distcc distributes compilation jobs across volunteer machines running\n"
"distccd.  Jobs that cannot be distributed, such as linking or \n"
"preprocessing are run locally.  distcc should be used with make's -jN\n"
"option to execute in parallel on several machines.\n",
    SYSCONFDIR,
    DISTCC_DEFAULT_PORT);
}



static void dcc_get_my_basename(char *argv[], char **myname)
{
    char *progname;
    
    /* Find our basename */
    if ((progname = strrchr(argv[0], '/')) != NULL)
        progname++;
    else
        progname = argv[0];

    *myname = progname;
}


/**
 * distcc client entry point.
 *
 * This is typically called by make in place of the real compiler.
 *
 * Performs basic setup and checks for distcc arguments, and then kicks of
 * dcc_build_somewhere().
 **/
int main(int argc, char **argv)
{
    int status, sg_level, tweaked_path = 0;
    char **compiler_args, *progname;
    int ret;

    atexit(dcc_cleanup_tempfiles);
    atexit(dcc_remove_state_file);

    dcc_set_trace_from_env();

    dcc_trace_version();
    
    dcc_get_my_basename(argv, &progname);

    /* Ignore SIGPIPE; we consistently check error codes and will
     * see the EPIPE. */
    dcc_ignore_sigpipe(1);

    sg_level = dcc_recursion_safeguard();

    if (strstr(progname, "distcc") != NULL) {
        /* Either "distcc -c hello.c" or "distcc gcc -c hello.c" */
        if (argc <= 1 || !strcmp(argv[1], "--help")) {
            dcc_show_usage();
            exit(0);
        }
        if (!strcmp(argv[1], "--version")) {
            dcc_show_version("distcc");
            exit(0);
        }
        dcc_find_compiler(argv, &compiler_args);
        dcc_get_my_basename(compiler_args, &progname);
        if ((ret = dcc_trim_path(progname)) != 0)
            dcc_exit(ret);
    } else {
        /* Invoked as "cc -c hello.c", with masqueraded path */
        if ((ret = dcc_support_masquerade(argv, progname, &tweaked_path)) != 0)
            dcc_exit(ret);
        
        dcc_shallowcopy_argv(argv, &compiler_args, 0);
        compiler_args[0] = progname;
    }

    if (sg_level - tweaked_path > 0) {
        rs_log_crit("distcc seems to have invoked itself recursively!");
        dcc_exit(EXIT_RECURSION);
    }

    dcc_exit(dcc_build_somewhere_timed(compiler_args, sg_level, &status));
}
