/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 *
 * Copyright (C) 2003 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include "config.h"

#ifdef HAVE_STDINT_H
#  include <stdint.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>

#include "types.h"
#include "distcc.h"
#include "rpc.h"
#include "io.h"
#include "trace.h"
#include "exitcode.h"
#include "state.h"
#include "tempfile.h"
#include "filename.h"
#include "snprintf.h"


static char *state_dir;

/* Use these strings for compiler state to keep things centralized.  They're
 * essentially tokens, not freeform strings. */
const char *STATE_COMPILE = "Compile",
    *STATE_CONNECT = "Connect",
    *STATE_SEND = "Send",
    *STATE_RECEIVE = "Receive",
    *STATE_BLOCKED = "Starved",
    *STATE_STARTUP = "Startup",
    *STATE_CPP = "Preprocessor";


const char *dcc_state_prefix = "state_";
const char *dcc_state_temp_prefix = "temp_";

/**
 * @file
 *
 * This file provides a way for distcc processes to make little notes
 * about what they're up to that can be read by a monitor process.
 *
 * State is stored as follows.
 *
 * Within our temporary directory, we create a subdirectory called "state".
 *
 * Each process creates a file named "state%d", for its pid.  This contains a
 * series of strings encoded by our standard network protocol.
 *
 * The files are created by the regular Unix trick of writing to a
 * temporary file and renaming into place, so the monitor should not
 * see any half-created files.
 *
 * Any process reading these files needs to handle the fact that they may be
 * truncated or otherwise incorrect.
 *
 * State files are deleted when the process is complete.
 *
 * Any files in here over a certain age (30s) are considered obsolete
 * and can be deleted by any process that notices them.
 *
 * The reader interface for these files is in mon.c
 *
 * @todo Perhaps arrange to delete the state file from a nonfatal
 * signal handler?  SIGINT in particular.
 **/


/**
 * Return a static pointer to the directory name for status files.
 *
 * The directory is created if it does not exist.
 **/
int dcc_get_state_dir (char **p)
{
    int ret;
    
    if (state_dir == NULL) {
        asprintf(&state_dir, "%s/state", dcc_get_tempdir ());
        if ((ret = dcc_make_dir (state_dir)))
            return ret;
    }

    *p = state_dir;

    return 0;
}


/**
 * Return newly allocated buffer holding the name of this process's state file.
 *
 * (This can't reliably be static because we might fork...)
 **/
static int dcc_get_state_filenames (char **fname, char **temp_fname)
{
    int ret;
    char *dir;
    
    if ((ret = dcc_get_state_dir(&dir)))
        return ret;

    asprintf(fname, "%s/%s%ld", dir, dcc_state_prefix, (long) getpid());
    if (temp_fname) {
        asprintf(temp_fname, "%s/%s%ld", dir, dcc_state_temp_prefix,
                 (long) getpid());
    }
    
    return 0;
}


/**
 * Get a file descriptor for writing to this process's state temp
 * file.
 **/
static int dcc_open_temp_state(int *p_fd,
                               const char *fname)
{
    int fd;

    fd = open(fname, O_CREAT|O_WRONLY|O_TRUNC|O_BINARY, 0666);
    if (fd == -1) {
        rs_log_error("failed to open %s: %s", fname, strerror(errno));
        return EXIT_IO_ERROR;
    }

    *p_fd = fd;
    return 0;
}


/**
 * Remove the state file for this process.
 *
 * This can be called from atexit().
 **/
void dcc_remove_state_file (void)
{
    char *fname;
    int ret;

    if ((ret = dcc_get_state_filenames(&fname, NULL)))
        return;

    if (unlink(fname) == -1) {
        /* It's OK if we never created it */
        if (errno != ENOENT) {
            rs_log_warning("failed to unlink %s: %s", fname, strerror(errno));
            ret = EXIT_IO_ERROR;
        }
    }

    free(fname);
}


static int dcc_write_state(int fd,
                           const char *state,
                           const char *file,
                           const char *host)
{
    int ret;

    if ((ret = dcc_x_token_int(fd, "STFV", 0)))
        return ret;

    if ((ret = dcc_x_token_int(fd, "CPID", (unsigned) getpid())))
        return ret;
    
    if ((ret = dcc_x_token_string(fd, "STTE", state)))
        return ret;

    if ((ret = dcc_x_token_string(fd, "FILE", file ? file : "")))
        return ret;

    if ((ret = dcc_x_token_string(fd, "HOST", host ? host : "")))
        return ret;

    return 0;
}


/**
 * Record the state of this process.
 *
 * The filename is trimmed down to its basename.
 *
 * If the source_file or host are NULL, then are left unchanged from
 * their previous value.
 **/
int dcc_note_state (const char *state,
                    const char *source_file,
                    const char *host)
{
    int fd;
    int ret;
    char *fname, *temp_name;
    static const char *last_fname, *last_host;

    if ((ret = dcc_get_state_filenames (&fname, &temp_name)))
        return ret;

    source_file = dcc_find_basename(source_file);
    if (source_file)
        last_fname = source_file;
    else
        source_file = last_fname;

    if (host)
        last_host = host;
    else
        host = last_host;

    rs_trace("note state %s, file \"%s\", host \"%s\"", state,
             source_file ? source_file : "(NULL)",
             host ? host : "(NULL)");

    if ((ret = dcc_open_temp_state(&fd, temp_name))) {
        free(fname);
        free(temp_name);
        return ret;
    }

    if ((ret = dcc_write_state(fd, state, source_file, host))) {
        dcc_close(fd);
        unlink(temp_name);
        free(temp_name);
        free(fname);
        return ret;
    }

    dcc_close(fd);

#ifdef __CYGWIN__
    /* can't rename over the top */
    unlink(fname);
#endif
    if (rename(temp_name, fname) == -1) {
        rs_log_error("rename %s to %s failed: %s", temp_name, fname,
                     strerror(errno));
        unlink(temp_name);
        free(temp_name);
        free(fname);
        return EXIT_IO_ERROR;
    }

    return 0;
}
