/* -*- c-file-style: "java"; indent-tabs-mode: nil -*-
 * 
 * distcc -- A simple distributed compiler system
 *
 * Copyright (C) 2002, 2003 by Martin Pool <mbp@samba.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */


                /* "More computing sins are committed in the name of
                 * efficiency (without necessarily achieving it) than
                 * for any other single reason - including blind
                 * stupidity."  -- W.A. Wulf
                 */



#include "config.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <errno.h>

#include <sys/stat.h>
#include <sys/types.h>

#include "distcc.h"
#include "trace.h"
#include "util.h"
#include "tempfile.h"
#include "snprintf.h"
#include "exitcode.h"



/**
 * @file
 *
 * Routines for naming, generating and removing temporary files and
 * fifos.
 *
 * All our temporary files are held in a single directory owned by the
 * user running the program.  I think this avoids any security or
 * symlink problems, and it keeps things kind of tidy.  I'm not
 * completely sure, though.
 *
 * It's fine for the files in that directory to be removed by the
 * tmpreaper, or indeed by hand if you're sure they're no longer
 * active.
 *
 * We also support generation of fifos (aka named pipes), which are
 * used for feeding data from the preprocessor and to the compiler.
 **/


/** A subdirectory of tmp to hold all our stuff.  NULL until
 * initialized. **/
static char *tempdir;

/**
 * A list of files that need to be cleaned up on exit.  The fixed-size
 * array is kind of cheap and nasty, but we're never going to use that
 * many.
 *
 * Volatile because it can be read from signal handlers.
 **/
#define N_CLEANUPS 20
volatile char *cleanups[N_CLEANUPS];

static const char *dcc_get_tmp_top(void)
{
    const char *d;

    d = getenv("TMPDIR");
    /* some sanity checks */
    if (!d || d[0] == '\0' || d[0] != '/' || d[1] == '\0') {
        return "/tmp";
    } else {
        return d;
    }
}



/**
 * Create a new fifo node, replacing whatever previously had the name.
 * The fifo is not opened.
 *
 * The fifo is opened without blocking to wait for somebody to start
 * reading; however the file descriptor is immediately set back to
 * blocking mode, so the first write to it will block.
 *
 * @param mode Permission bits for new fifo.
 *
 * @returns 0 for OK, or error.
 **/
int dcc_mkfifo(const char *fifo_name)
{
    /* Just try to unlink once; don't worry if it doesn't exist.  This is to
     * handle the occasional temporary file which might get left behind from a
     * previous invocation with the same pid. */
    
    if (dcc_remove_if_exists(fifo_name))
        return EXIT_IO_ERROR;

    if (mkfifo(fifo_name, S_IRUSR|S_IWUSR) == -1) {
        rs_log_warning("failed to make fifo %s: %s", fifo_name,
                       strerror(errno));
        return EXIT_IO_ERROR;
    }

    return 0;
}



/**
 * Create the directory @p path.  If it already exists as a directory we succeed.
 **/
int dcc_make_dir (const char *path)
{
    struct stat buf;

    if (mkdir(path, 0755) == 0) {
        return 0;               /* great */
    } else if (errno == EEXIST) {
        /* If there's already a directory of this name, then we will
         * have just failed with EEXIST.  We need to make sure that if
         * there was an existing file, then it is actually a
         * directory.
         *
         * This protects against people causing mischief by creating
         * the directory as a symlink to somewhere else. */
        if (lstat(path, &buf) == -1) {
            rs_log_error("lstat %s failed: %s", path, strerror(errno));
            return EXIT_IO_ERROR;
        } else if (!S_ISDIR(buf.st_mode)) {
            rs_log_error("%s is not a directory", path);
            return EXIT_IO_ERROR;
        } else if (buf.st_mode & (S_IWGRP | S_IWOTH)) {
            rs_log_error("permissions on %s (%#o) seem insecure", path,
                         (int) buf.st_mode);
            return EXIT_IO_ERROR;
        } else {
            return 0;
        }
    } else {
        rs_log_error("mkdir %s failed: %s", path, strerror(errno));
        return EXIT_IO_ERROR;
    }
}


/**
 * Create a temporary directory used for all files for this user and
 * host.
 *
 * We uniquify by hostname because otherwise pids won't be unique, and
 * they might cause intermittent failures if you have a shared /tmp.
 * (Which sounds like a dumb idea, but...)
 **/
int dcc_setup_tempdir(void)
{
    const char *tmp;
    char host[256];

    tmp = dcc_get_tmp_top();

    if (gethostname(host, sizeof host) == -1) {
        strcpy(host, "unknown");
    }

    /* We don't want to look up the username here because that might
     * be slow, and it's pretty unnecessary.  We can't trust $USER
     * because that can easily cause trouble when we're started with
     * --user or through sudo.  The daemon would likely try to write
     * to root's temp directory and fail. */
    asprintf(&tempdir, "%s/distcc_%s_%ld", tmp, host, (long) getuid());

    return dcc_make_dir (tempdir);
}


char *dcc_make_tmpnam(const char *prefix, const char *suffix)
{
    char *s;
    int i;

    if (!tempdir)
        dcc_setup_tempdir();

    for (i = 0; cleanups[i]; i++)
        ;

    if (i >= N_CLEANUPS) {
        rs_fatal("too many cleanups");
    }

    asprintf(&s, "%s/%s_%ld%s", tempdir, prefix,
             (long) getpid() & 0xffffffffUL, suffix);

    cleanups[i] = s;
    
    return s;
}


const char *dcc_get_tempdir(void)
{
    if (!tempdir)
        dcc_setup_tempdir();
    return tempdir;
}


/**
 * The daemon runs from inside the temporary directory.
 **/
int dcc_enter_tempdir(void)
{
    if (chdir(dcc_get_tempdir()) == -1) {
        rs_log_error("failed to change to %s: %s", dcc_get_tempdir(),
                     strerror(errno));
        return EXIT_IO_ERROR;
    }
    return 0;
}


/**
 * You can call this at any time, or hook it into atexit()
 *
 * If $DISTCC_SAVE_TEMPS is set to "1", then files are not actually
 * deleted -- good for debugging.
 **/
void dcc_cleanup_tempfiles(void)
{
    int i;
    int done = 0;

    if (dcc_getenv_bool("DISTCC_SAVE_TEMPS", 0)) /* tempus fugit */
        return;

    for (i = 0; i < N_CLEANUPS && cleanups[i]; i++) {
        if (unlink((char *) cleanups[i]) == -1 && (errno != ENOENT)) {
            rs_log_notice("cleanup %s failed: %s", cleanups[i],
                          strerror(errno));
        }
        done++;
        free((char *) cleanups[i]);
        cleanups[i] = NULL;
    }

    rs_trace("deleted %d temporary files", done);
}
