/* --------------------------------- vesa.c --------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* VGA level video control functions.
*/

#include "fly.h"
#include "vesa.h"

#include <conio.h>
#include <dos.h>



Ulong	vesa_BankSwitches = 0;

static int	far vesa_page (Ulong base);
static void	far vesa_bank (int bank);

static Uint	Width = 640;

static void (far *VesaSwitch) (void) = NULL;
static int	use_vga = 1;		/* means SVGA not available */

typedef struct {
	Uint	ax;
	Uint	bx;
	Uint	cx;
	Uint	dx;
	Uint	si;
	Uint	di;
	Uint	bp;
	Uint	f;
	Uint	ds;
	Uint	es;
} REGISTERS;

#define VGA_PAGE	((Uchar far *)0xa0000000L)
#define MY_OFF(p)	(((Ushort far *)&(p))[0])
#define MY_SEG(p)	(((Ushort far *)&(p))[1])

static int far
vesa_int10 (REGISTERS *regs)
{
	union REGS	inregs, outregs;
	struct SREGS    sregs;

	inregs.x.ax = regs->ax;
	inregs.x.bx = regs->bx;
	inregs.x.cx = regs->cx;
	inregs.x.dx = regs->dx;
	inregs.x.si = regs->si;
	inregs.x.di = regs->di;
	sregs.ds = regs->ds;
	sregs.es = regs->es;

	(void) int86x (0x10,&inregs,&outregs,&sregs);

	regs->ax = outregs.x.ax;
	regs->bx = outregs.x.bx;
	regs->cx = outregs.x.cx;
	regs->dx = outregs.x.dx;
	regs->si = outregs.x.si;
	regs->di = outregs.x.di;
	regs->ds = sregs.ds;
	regs->es = sregs.es;

	return (outregs.x.cflag);
}

extern int far
vesa_page (Ulong base)			/* Set visual address */
{
	REGISTERS	regs;

	while (inp (0x3da) & 0x01)	/* wait for Display Enabled */
		;

	if (use_vga) {
		base >>= 2;
		outp (0x3D4, 0x0d);
		outp (0x3D5, (Uchar)(base & 0x0ff));
		outp (0x3D4, 0x0c);
		outp (0x3D5, (Uchar)((base >> 8) & 0x0f));
	} else {
		regs.dx = (Uint)(base / Width);
		regs.cx = (Uint)(base % Width);
	        regs.ax = 0x4F07;
	        regs.bx = 0;
	        vesa_int10 (&regs);
	}


	while (inp (0x3da) & 0x08)	/* wait for Vert Sync*/
		;
	while (!(inp (0x3da) & 0x08))	/* wait for Vert Sync end */
		;
	return (0);
}

extern void far
vesa_bank (int bank)			/* set active (64KB) page */
{
	Ushort	flags;
	static int	oldbank = 0;

	if (oldbank != bank && VesaSwitch) {
		oldbank = bank;
		++vesa_BankSwitches;

		bank *= 1;		/* granularity */

_asm {
		pushf
		pop flags
		cli

		mov dx,bank
		mov bx,0
		call DWORD PTR [VesaSwitch]

		mov dx,bank
		mov bx,1
		call DWORD PTR [VesaSwitch]

		push flags
		popf
}
	}
}

extern void far
vesa_palette (int color, int r, int g, int b)
{
	short	flags;

_asm {
	pushf
	pop flags
	cli
}

	outp (0x3c8, color);	(void)inp (0x080);
	outp (0x3c9, r >> 2);	(void)inp (0x080);
	outp (0x3c9, g >> 2);	(void)inp (0x080);
	outp (0x3c9, b >> 2);
_asm {
	push flags
	popf
}
}

static Uchar	far infomem[256] = {0};

extern void far
vesa_init (Uint mode, int width, int height, int xbytes)
{
	REGISTERS	regs;
	Uchar		far *info = infomem;

	VesaSwitch = NULL;

	regs.ax = 0x4F02;		/* set mode */
	regs.bx = mode;
	vesa_int10 (&regs);

	regs.ax = 0x4F01;		/* get info */
	regs.cx = mode;
	regs.es = MY_SEG (info);
	regs.di = MY_OFF (info);
	vesa_int10 (&regs);
	if (0x004F == regs.ax)
		memcpy ((char far*)&VesaSwitch, info+12, sizeof (VesaSwitch));

	Width = width;
	use_vga = 0;
	vesa_page (0x40000UL);		/* set visual address */

        regs.ax = 0x4F07;		/* Read back address */
	regs.bx = 1;
	vesa_int10 (&regs);

	if (0x004F != regs.ax ||
	    0 != (regs.bx & 0xff00) ||
	    0x40000UL != (regs.dx * (Ulong)Width + regs.cx))
		use_vga = 1;
	vesa_page (0UL);
	vInit (VGA_PAGE, width, height, xbytes, vesa_bank, vesa_page);
}

extern void far
vesa_term (void)
{
	REGISTERS	regs;

	regs.ax = 0x03;		/* text mode */
	vesa_int10 (&regs);
}



/* A sample main() to demostrate the usage of the functions.
*/



Ulong	x = 0x76859403UL;

static void
pause (int i)
{
	int	j;

	for (; i > 0; --i)
		for (j = 1000; j > 0; --j)
			x = x * 0x12345678UL;
}

int
main (int argc, char *argv[])
{
	int	mode, width, height;
	int	p, x, y, rx, ry, c;

/* parse parameters.
*/
	if (argc != 2) {
		fprintf (stderr, "usage:\tvesa mode\n");
		fprintf (stderr, "\tmode is 640,800,1024 or1280\n");
		exit (1);
	}

	if (!strcmp ("640", argv[1])) {
		mode = 1;
		width = 640;
		height = 480;
	} else if (!strcmp ("800", argv[1])) {
		mode = 3;
		width = 800;
		height = 600;
	} else if (!strcmp ("1024", argv[1])) {
		mode = 5;
		width = 1024;
		height = 768;
	} else if (!strcmp ("1280", argv[1])) {
		mode = 7;
		width = 1280;
		height = 1024;
	} else {
		fprintf (stderr, "unrecognized parameter\n");
		exit (1);
	}
		
/* first set the vesa system.
*/
	vesa_init (0x100+mode, width, height, width);

/* wait for the monitor to stabilise.
*/
	pause (400);
	vSetWriteMode (T_MXOR);

/* squares in first page.
*/
	vSetActive (0);
	for (p = 0; p < 2*height; ++p) {
		x = rand () % (width-50);
		y = rand () % (height-50);
		c = rand () % 16;
		vMoveTo (   x,    y);
		vDrawTo (50+x,    y, c);
		vDrawTo (50+x, 50+y, c);
		vDrawTo (   x, 50+y, c);
		vDrawTo (   x,    y, c);
		pause (1);
	}
	pause (100);

/* scroll up.
*/
	for (p = 0; p <= height; p += 4)
		vesa_page (p*(Ulong)width);

/* circles in second page.
*/
	vSetActive (1);
	for (p = 0; p < 2*height; ++p) {
		x = rand () % (width-50);
		y = rand () % (height-50);
		rx = rand () % 21 + 5;
		ry = rand () % 21 + 5;
		c = rand () % 16;
		vEllipse (x+25, y+25, rx, ry, c);
		pause (1);
	}
	pause (100);

/* scroll back.
*/
	for (p = height; (p -= 4) >= 0;)
		vesa_page (p*(Ulong)width);
	pause (100);

	vesa_term ();

	printf ("use_vga      %u\n", use_vga);
	printf ("BankSwitches %lu\n", vesa_BankSwitches);

	bLogStats ();

	exit (0);
	return (0);
}
