/*
** Display.c : This modules handles the on-screen display for the user
*/
#include "game.h"

/*
** RedrawMap(x,y) clears the playing area and draws a new map, centred
** on the pixel coordinates (x,y).
**
*/

static int MX,MY;		/* Centre coords of map (pixels) */

void ScrollMapCentre(int dx, int dy)
	{
	MX = MX+dx;
	MY = MY+dy;
	}

void SetMapCentre(int x, int y)
	{
	MX=x; MY=y; 
	InvalidateALLPlates();
	}

void GetMapCentre(int *x, int *y) { *x = MX; *y = MY; }

int
RedrawMap()
	{
	int i,j,x,y,dx,dy;
	int nj,ni,cx,cy;

	/* Calculate pixel offset from cell boundary
	** (-dx,-dy) is also the starting screen coordinate for drawing
	*/
	dx = MX & (CELL_SIZE - 1); dy = MY & (CELL_SIZE - 1);

	/* Determine the starting cell for drawing */
	cx = (MX - MAP_WIDTH/2) / CELL_SIZE;
	cy = (MY - MAP_HEIGHT/2) / CELL_SIZE;

	for(j = cy, nj=0, y=-dy; nj != MAP_CELLS_DOWN+1; ++nj, ++j, y += CELL_SIZE)
	   for(i = cx,ni=0, x=-dx; ni != MAP_CELLS_ACROSS+1; ++ni, ++i, x += CELL_SIZE)
		DrawCell(j,i,y,x);
	return 0;
	}

/*
** This routine takes a cell (cx,cy) and calculates the screen coordinates
** corresponding to the top left corner, returning these in (*x,*y).
** We need to know the map centre coordinates for this (MX,MY).
**
** This routine can return coords which are not within the visible area.
*/
ScreenCoords(int cx, int cy, int *x, int *y)
	{
	int x1,y1;

	cx *= CELL_SIZE; cy *= CELL_SIZE;

	x1 = cx - MX;
	y1 = cy - MY;
	*x = MAP_WIDTH/2 +  x1;
	*y = MAP_HEIGHT/2 + y1;
	return 0;
	}

/*
** Keep a list of areas on the screen which require redrawing due to movement
** of sprites etc.
**
*/
typedef struct
	{
	short x1,y1,x2,y2;		/* rectangular area */
	} Region;

#define MAXREGIONS	1024

static Region RList[MAXREGIONS];
static int nRegions = 0;

void
InvalidateRegion(int x1, int y1, int x2, int y2)
	{
	int i,area,px1,py1,px2,py2;
	Region *r;

	/*
	** Expand our region out to a multiple of CELL_SIZE
	*/
	x1 -= (x1&(CELL_SIZE-1));
	y1 -= (y1&(CELL_SIZE-1));
	x2 -= (x2&(CELL_SIZE-1)); x2+=CELL_SIZE-1;
	y2 -= (y2&(CELL_SIZE-1)); y2+=CELL_SIZE-1;

	if (nRegions >= MAXREGIONS-1) Die("InvalidateRegion: Too many regions");
	area = (x2-x1+1) * (y2-y1+1);
	for(i=0; i<nRegions; ++i)
	   {
	   r = &RList[i];
	   /*
	   ** Is this region contained wholly inside another ?
	   */
	   if (x1 >= r->x1 && y1 >= r->y1 && x2 <= r->x2 && y2 <= r->y2)
		return;

	   /*
	   ** Does the new region wholly encompass an existing one ?
	   */
	   if (x1<=r->x1 && y1<=r->y1 && x2>=r->x2 && y2>=r->y2)
		{ r->x1=x1; r->y1=y1; r->x2=x2; r->y2=y2; return;}

	   /*
	   ** Check for partial overlaps which add up to complete overlap
	   */
	   if (Overlap_c(x1,y1,x2,y2,r->x1,r->y1,r->x2,r->y2,&px1,&py1,&px2,&py2))
		{
		area -= (px2-px1+1) * (py2-py1+1);
		if (area==0) return;
		if (area<0) Die("Invalidateregion: area too large");
		}
	   }

	/*
	** This is a new region, add it to the list
	*/
	r = &RList[nRegions++];
	r->x1=x1; r->y1=y1; r->x2=x2; r->y2=y2;
	}

void
ResetRegions() { nRegions=0; }

void
RedrawRegions()
	{
	int i;
	Region *r;

	for(i=0; i<nRegions; ++i)
	   {
	   r = &RList[i];
	   /*
	   ** Redraw the underlying terrain
	   */
	   RedrawMapSubSection(r->x1,r->y1,r->x2,r->y2);

	   UpdateEntities(r->x1,r->y1,r->x2,r->y2);
	   }
	nRegions=0;
	}
