#include "game.h"

#ifdef USE_JUGL
#include <jugl.h>

static struct SC_display *sc_dpy;
static struct SC_frame *sc_frm;
#else
Display *dpy = NULL;
static int scr;
static int Win=0;
#endif

static int scr;

/* All the different ways of accessing memory */
static unsigned short *smem = NULL;
static unsigned int   *imem = NULL;

int _GrLineOffsetB=0;	/* Line offset in bytes, words, long & quad */
int _GrLineOffsetW=0;
int _GrLineOffsetL=0;
int _GrLineOffsetQ=0;

static int bpp = 2;

int _GrXMIN=0, _GrXMAX=0, _GrYMIN=0, _GrYMAX=0;

#ifdef USE_JUGL
int
GrInit(int wid, int hgt, int depth)
	{
	int i;
	char name[5];

printf("JUGL %s %d\n",__FILE__, __LINE__); fflush(stdout);
	sc_dpy = SC_open_display("Altair",wid,hgt,depth);

	if (sc_dpy == NULL) Die("Could not open display");
	GrNextFrame();
	return 0;
	}

void
GrKeycode(int *code, char *str)
	{
	SC_get_keycode_from_keyname(code,str);
	}

void
GrShowFrame()
	{
	SC_show_frame(sc_dpy,sc_frm);
	}

void
GrNextFrame()
	{
	sc_frm = SC_next_frame(sc_dpy);
	if (sc_frm == NULL) Die("GrShowPage: SC_next_frame: Error");

	_GrLineOffsetB = sc_frm->bytes_per_line;
	_GrLineOffsetW = _GrLineOffsetB >> 1;
	_GrLineOffsetL = _GrLineOffsetW >> 1;
	_GrLineOffsetQ = _GrLineOffsetL >> 1;

	smem = (unsigned short *)sc_frm->pixels;
	imem = (unsigned int *)smem;
	}

void
GrShutdown()
	{
	SC_close_display(sc_dpy);
	}

int
GrKeyPress(int *ispress)
	{
	int i,key;

	if (SC_get_key(sc_dpy,&key,ispress))
	  return key;

	return 0;
	}

int
GrWaitforKeyPress()
	{
	int key,press;

	while(1)
	   {
	   while ((key=GrKeyPress(&press)) == 0);
	   if (press) return key;
	   }

	return 0;
	}
#else
int
GrInit(int wid, int hgt, int depth)
	{
	int i,fd,flags,err,Width,banksz,memsz,dotclock;
	XSetWindowAttributes xw;
	XF86VidModeModeLine vmode;
	Visual *vis;
	GC gc;

	dpy = XOpenDisplay(":0");
	if (dpy == NULL) Die("Could not open display");
	scr = DefaultScreen(dpy);

	if (wid<=0 || hgt<=0 || (wid&7))
	   Die("GrInit: Invalid dimensions");

	if (DefaultDepth(dpy,scr) != depth)
	   Die("GrInit: Display depth must be %d",depth);

	flags=0;
	err = XF86DGAQueryDirectVideo(dpy,scr,&flags);
	if (!(flags & XF86DGADirectPresent))
	   Die("GrInit: DGA Not Supported on this video card");

	/*
	** Check for read/write permission on /dev/mem
	*/
	if (access("/dev/mem",R_OK) || access("/dev/mem",W_OK))
	   Die("GrInit: Must have read/write access on /dev/mem (run as root?)");

	/********************************************************************/

	/*************************************************
	** Create a window that covers the whole screen **
	*************************************************/

	/* XF86VidModeGetModeLine(dpy,scr,&dotclock,&vmode); */
	vmode.hdisplay = 1024;
	vmode.vdisplay = 768;
	

	vis = DefaultVisual(dpy,scr);
	gc  = XDefaultGC(dpy,scr);
	xw.override_redirect = True;
	xw.event_mask = ButtonPressMask|PointerMotionMask|KeyPressMask |
			ButtonReleaseMask|KeyReleaseMask;
	Win = XCreateWindow(dpy,DefaultRootWindow(dpy),0,0,vmode.hdisplay,vmode.vdisplay,
		0, depth, InputOutput, vis,
		CWEventMask | CWOverrideRedirect, &xw);
printf("%s %d\n",__FILE__, __LINE__); fflush(stdout);
	XMapWindow(dpy,Win);
printf("%s %d\n",__FILE__, __LINE__); fflush(stdout);
	XRaiseWindow(dpy,Win);
printf("%s %d\n",__FILE__, __LINE__); fflush(stdout);
	/********************************************************************/
	
	/* Grab the pointer and keyboard */
	XGrabKeyboard(dpy, Win, True, GrabModeAsync, GrabModeAsync, CurrentTime);
printf("%s %d\n",__FILE__, __LINE__); fflush(stdout);
	XGrabPointer(dpy, Win, True,
		ButtonPressMask|ButtonReleaseMask|PointerMotionMask,
		GrabModeAsync,GrabModeAsync,None,None,CurrentTime);

printf("%s %d\n",__FILE__, __LINE__); fflush(stdout);
	/********************************************************************/

	flags = XF86DGADirectGraphics | XF86DGADirectKeyb | XF86DGADirectMouse;
printf("%s %d\n",__FILE__, __LINE__); fflush(stdout);
	err = XF86DGADirectVideo(dpy,scr,flags);
	if (!err) Die("GrInit: Error initialising DGA");
printf("%s %d\n",__FILE__, __LINE__); fflush(stdout);

	XF86DGASetViewPort(dpy,scr,0,0);
printf("%s %d\n",__FILE__, __LINE__); fflush(stdout);
	XF86DGAGetVideo(dpy,scr,(char **)&smem,&_GrLineOffsetW,&banksz,&memsz);

printf("%s %d\n",__FILE__, __LINE__); fflush(stdout);
	_GrLineOffsetB = _GrLineOffsetW<<1;
	_GrLineOffsetL = _GrLineOffsetW>>1;
	_GrLineOffsetQ = _GrLineOffsetL>>1;

	/* Is the screen wide enough ? */
	if (_GrLineOffsetW < wid)
	   Die("GrInit: Screen is not wide enough for requested size");

	memsz *= 1024;

	/* How many lines on this display? */
	if (memsz / (_GrLineOffsetB) < hgt)
	   Die("GrInit: Screen does nt have enough lines for requested size");

	_GrXMIN = _GrYMIN=0;
	_GrXMAX=wid-1; _GrYMAX=hgt-1;
	/*
	** Check that this is not bank switched
	*/
	if (banksz != memsz) 
	   Die("GrInit: Banked video modes not supported.");

	/*
	** Allow writing to it
	*/
	if (mprotect((caddr_t)smem,memsz,PROT_READ|PROT_WRITE))
	   Die("GrInit: Cannot set memory region read/write");
	imem=(unsigned int *)smem;

	/* Set for non-blocking I/O */
	system("stty -echo raw");
	flags = fcntl(0,F_GETFL,0);
	flags |= O_NONBLOCK;
	fcntl(0,F_SETFL,flags);

	return(0);
	}

void
GrKeycode(int *code, char *name)
	{
	*code=0;
	if (name[0] && name[1]==0) *code = name[0];
	if (!strcmp(name,"esc")) *code = ESC;
	if (!strcmp(name,"space")) *code = ' ';
	}

void
GrShutdown()
	{
printf("Doing shutdown\n");
	if (dpy == NULL) return;
	XF86DGADirectVideo(dpy,scr,0);
	imem = NULL; smem = NULL;
	_GrXMIN = _GrYMIN = _GrXMAX = _GrYMAX=0;
	system("stty sane");
	}

int
GrKeyPress(int *ispress)
	{
	int k;


	k = getchar();
	if (k>0) *ispress=1;
	return k;
	}

int
GrWaitforKeyPress()
	{
	int ch;

	while((ch=getchar()) <=0);
	return ch;
	}
#endif

void
GrSetDrawingRegion(int x1, int y1, int x2, int y2)
	{
	_GrXMIN=x1; _GrYMIN=y1;
	_GrXMAX = x2;
	_GrYMAX = y2;
	}

GrObject *
GrCreateObject(int width, int height)
	{
	GrObject *g;

	g = (GrObject *) malloc(sizeof(GrObject));
	if (g==NULL) Die("GrCreateObject: Out of memory");

	g->name = NULL;
	g->Width = width;
	g->Height = height;
	g->PixelsPerLine = width;	/* 2 bytes per pixel */
	g->mapped=0;

	g->mem = (char *)malloc(width * height * bpp + 7);
	if (g->mem==NULL) Die("GrCreateObject: Out of memory");

	/* Align our data region to start on 8-byte boundary */
	g->data = (ushort *)(((uint)g->mem+7) & (~7));

	return g;
	}

void
GrDestroyObject(GrObject *g)
	{
	if (g==NULL) Die("GrDestroyObject: Passed NULL pointer");
	if (g->mem) free(g->mem);
	free(g);
	}

/*****************************************************************************
**                                                                           *
**  GrMergeObject: Routines for combining objects                            *
**                                                                           *
*****************************************************************************/

/*
** Fill an object with a solid colour
*/
void
GrFillObject(GrObject *dst, ushort cval)
	{
	int x = dst->Width * dst->Height;
	ushort *d = dst->data;

	while(x--) *(d++) = cval;
	}

void
GrMergeObject(GrObject *gd, GrObject *gs, int dx, int dy, int op)
	{
	int dx1,dy1,dx2,dy2;
	int xmax,ymax;

	/* Check that we are inside the bounds of the target object */	
	xmax = gd->Width-1; ymax = gd->Height-1;
	dx1 = Clip1D(dx,0,xmax); dy1=Clip1D(dy,0,ymax);
	dx2 = Clip1D(dx1+gs->Width-1,0,xmax);
	dy1=Clip1D(dy1+gs->Height-1,0,ymax);

	if (dx2-dx1<=0 || dy2-dy1<=0) return;
	GrMergeObjectSubSectionNC(gd,gs,dx1,dy1, dx1-dx, dy1-dy,
		(dx1-dx) + (dx2-dx1), (dy1-dy) + (dy2-dy1), op);
	}

void
GrMergeObjectNC(GrObject *gd, GrObject *gs, int dx, int dy, int op)
	{
	GrMergeObjectSubSectionNC(gd,gs,dx,dy,0,0,gs->Width-1,gs->Height-1,op);
	}

void
GrMergeObjectSubSection(GrObject *gd, GrObject *gs, int dx, int dy,
			int x1, int y1, int x2, int y2, int op)
	{
	int xmax,ymax;
	int dx1,dy1,dx2,dy2;

	/* Clip the source rectangle against the source object */
	xmax = gs->Width-1; ymax = gs->Height-1;
	x1 = Clip1D(x1,0,xmax); x2 = Clip1D(x2,0,xmax);
	y1 = Clip1D(y1,0,ymax); y2 = Clip1D(y2,0,ymax);
	if (y2-y1<=0 || x2-x1<=0) return;

	/* Clip the source dimensions against the destination object */
	xmax = gd->Width-1; ymax = gd->Height-1;
	dx1 = Clip1D(dx,0,xmax); dy1=Clip1D(dy,0,ymax);
	dx2 = Clip1D(dx1+(x2-x1),0,xmax);
	dy1=Clip1D(dy1+(y2-y1),0,ymax);

	if (dx2-dx1<=0 || dy2-dy1<=0) return;
	GrMergeObjectSubSectionNC(gd,gs,dx1,dy1,x1,y1,x1+(dx2-dx1),y1+(dy2-dy1), op);
	}

void
GrMergeObjectSubSectionNC(GrObject *gd, GrObject *gs, int dx, int dy,
			int x1, int y1, int x2, int y2, int op)
	{
	register uint i,val;
	register ushort *dst,*src;
	int dst_inc,src_inc,x,y,w,h;

	w = x2-x1+1; h = y2-y1+1;

	dst = (ushort *) &gd->data[gd->PixelsPerLine*dy + dx];
	src = (ushort *) &gs->data[gs->PixelsPerLine*y1 + x1];
	src_inc = gs->PixelsPerLine - w;
	dst_inc = gd->PixelsPerLine - w;

	while(h--)
	   {
	   switch(op)
		{
		case GrWRITE: for(i=w; i; --i) *(dst++) = *(src++); break;
		case GrAND: for(i=w; i; --i) *(dst++) &= *(src++); break;
		case GrOR: for(i=w; i; --i) *(dst++) |= *(src++); break;
		case GrXOR: for(i=w; i; --i) *(dst++) ^= *(src++); break;
		case GrIMAGE: 

			    for(i=w>>1; i; --i)
				{
				val=*((uint *)src);
				if ((val & TRANSPARENT_BIT_2PIXELS)==0)
				   *((uint *)dst) = val;
				else if ((val & TRANSPARENT_BIT)==0)
				   *(((ushort *)dst)) = (ushort)(val&0xffff);
				else if ((val&TRANSPARENT_BIT_HIGH)==0)
				   *(((ushort *)dst)+1) = (ushort)(val>>16);

				dst+=2; src+=2;
				}
			      if (w&1)
				{
				if ((*src & TRANSPARENT_BIT)==0) *dst = *src;
				++dst; ++src;
				}
			      break;
		}
	   dst += dst_inc; src += src_inc;
	   }
	}

/*****************************************************************************
**                                                                           *
**  GrDrawObject Routines for displaying an object onto the screen.          *
**                                                                           *
*****************************************************************************/

void
GrDrawObject(GrObject *g, int x, int y)
	{
	int x1,y1,x2,y2;

	x1 = Clip1D(x,_GrXMIN,_GrXMAX);
	x2 = Clip1D(x+g->Width-1,_GrXMIN,_GrXMAX);
	y1 = Clip1D(y,_GrYMIN,_GrYMAX);
	y2 = Clip1D(y+g->Height-1,_GrYMIN,_GrYMAX);
	if (x1==x2 || y1==y2) return;

	GrDrawObjectSubSection(g,x1,y1,x1-x, y1-y,
		(x1-x) + (x2-x1), (y1-y) + (y2-y1));
	}

void
GrDrawObjectNC(GrObject *g, int x, int y)
	{
	GrDrawObjectSubSectionNC(g,x,y,0,0,g->Width-1,g->Height-1);
	}

void
GrDrawObjectSubSection(GrObject *g, int dx, int dy,int x1, int y1, int x2, int y2)
	{
	int xmax,ymax;
	int dx1,dy1,dx2,dy2;

	/* First step - clip internal coords against object dimensions */
	xmax = g->Width-1; ymax = g->Height-1;
	x1 = Clip1D(x1,0,xmax); x2 = Clip1D(x2,0,xmax);
	y1 = Clip1D(y1,0,ymax); y2 = Clip1D(y2,0,ymax);
	if (y2-y1<=0 || x2-x1<=0) return;

	/* Clip the target coords on the screen */
	dx1 = Clip1D(dx,_GrXMIN,_GrXMAX); dy1 = Clip1D(dy,_GrYMIN,_GrYMAX);
	dx2 = Clip1D(dx+(x2-x1),_GrXMIN,_GrXMAX);
	dy2 = Clip1D(dy+(y2-y1),_GrYMIN,_GrYMAX);
	if (dx2-dx1<=0 || dy2-dy1<=0) return;

	GrDrawObjectSubSectionNC(g,dx1,dy1,x1,y1,x1+(dx2-dx1),y1+(dy2-dy1));
	}

void
GrDrawObjectSubSectionNC(GrObject *g, int dx, int dy,int x1, int y1, int x2, int y2)
	{
	volatile int dst = (int)(&smem[_GrLineOffsetW*dy + dx]);
	volatile int src = (int)&g->data[g->PixelsPerLine*y1 + x1];
	volatile int w = (x2-x1+1) * bpp;
	volatile int h = y2-y1+1;
	volatile int src_inc,dst_inc;

	src_inc = g->PixelsPerLine*bpp - w;
	dst_inc = _GrLineOffsetB - w;

	if (g==NULL || smem==NULL) return;
	if (w<=0 || h<=0) return;

	/*
	** dst = Byte offset to destination (assume word aligned)
	** src = Byte offset to src (assume word aligned)
	** w   = Width of row in Bytes
	** h   = Number of rows
	** src_inc = Increment from end of one row to start of next in bytes
	** dst_inc = Increment from end of one row to start of next in bytes
	*/

	__asm__("
	pushal
	movl %0,%%edi
	movl %1,%%esi
	movl %3,%%ebx
	movl %4,%%edx
0:
	movl %2,%%eax
	cmpl $8,%%eax
	jl   3f
	test $4,%%edi
	je   3f
	movsl
	sub  $4,%%eax
3:
	movl %%eax,%%ecx
	and  $7,%%eax
	shr  $3,%%ecx
	je   2f
.align 32
1:
	fldl  (%%esi)
	fstpl (%%edi)
	add    $8,%%esi
	add    $8,%%edi
	dec    %%ecx
	jne    1b
2:
	movl %%eax,%%ecx
	rep
	movsb
	addl  %%edx,%%esi
	addl  %5,%%edi
	sub  $1,%%ebx
	jne  0b
	popal
	  " :: "g"(dst), "g"(src), "g"(w),
		"g"(h), "g"(src_inc), "g"(dst_inc));

	g->mapped=1;
	g->x = x1; g->y = y1;
	}

/*****************************************************************************
**                                                                           *
**  Miscellaneous drawing routines                                           *
**                                                                           *
*****************************************************************************/


void
GrSolidFill(int x1,int y1,int x2,int y2,unsigned short cval)
	{
	x1 = Clip1D(x1,_GrXMIN,_GrXMAX); x2 = Clip1D(x2,_GrXMIN,_GrXMAX);
	y1 = Clip1D(y1,_GrYMIN,_GrYMAX); y2 = Clip1D(y2,_GrYMIN,_GrYMAX);

	if (y2-y1<=0 || x2-x1<=0) return;
	GrSolidFillNC(x1,y1,x2,y2,cval);
	}

void
GrSolidFillNC(int x1,int y1,int x2,int y2,unsigned short cval)
	{
	int i,w,h;
	int o,inc;

	w = x2-x1+1; h=y2-y1+1;
	o = y1 * _GrLineOffsetW + x1;
	inc = _GrLineOffsetW - w;

	while(h--)
	   {
	   for(i=w; i; --i) smem[o++] = cval;
	   o += inc;
	   }
	}

void
GrScreenPlot(int x, int y, int val)
	{
	if (smem==NULL) return;
	if (x>=_GrXMIN && x<=_GrXMAX && y>=_GrYMIN && y<=_GrYMAX)
	   smem[_GrLineOffsetW * y + x] = val;
	}

void
GrScreenPlotNC(int x, int y, int val)
	{
	if (smem==NULL) return;

	smem[_GrLineOffsetW * y + x] = val;
	}
