/*
** This is the graphics module. It provides the interface to GRX20 and/or
** other Graphical API's
*/
#include "game.h"

typedef struct
	{
	unsigned int d[8];	/* We need 256 bits of storage */
	} TerrainSelector;

static int LocateSurroundingTile(int cx, int cy, TerrainSelector *s);
static int InitTiles(void);
static int InitColours(void);
static void ClearTerrainSelector(TerrainSelector *s);
static void InvertTerrainSelector(TerrainSelector *s);
static void SetTerrainSelector(TerrainSelector *s, int val);
static void UnSetTerrainSelector(TerrainSelector *s, int val);
static int TestTerrainSelector(TerrainSelector *s, int val);

/*
** Descriptors to keep track of the 16x16 tile sets used to
** draw the terrain.
*/
static GrObject *SystemTiles;
static GrObject *VarTiles[256];
static GrObject *LoadTileSet(char *);

GrObject *MapContext;

/****************************************************************************
**
**  Terrain Selectors are used to pass a set of possible terrain types
**  between functions. Since there is a max of 256 terrain types, we can
** use an array of 8 unsigned ints to hold all the bits.
**
****************************************************************************/

static void
ClearTerrainSelector(TerrainSelector *s)
	{
	int i;
	for(i=0; i<8; ++i) s->d[i]=0;
	}

static void
InvertTerrainSelector(TerrainSelector *s)
	{
	int i;
	for(i=0; i<8; ++i) s->d[i] ^= 0xffffffff;
	}

static void
SetTerrainSelector(TerrainSelector *s, int val)
	{  s->d[val>>5] |= (1 << (val&31)); }

static void
UnSetTerrainSelector(TerrainSelector *s, int val)
	{  s->d[val>>5] &= ~(1<<(val&31)); }

static int
TestTerrainSelector(TerrainSelector *s, int val)
	{ return ((s->d[val>>5] & (1 << (val&31))) ? 1: 0);}

/**************************************************************************/

int
InitGraphics()
	{
	int i;

	GrInit(SCREEN_WIDTH,SCREEN_HEIGHT,SCREEN_DEPTH);
	InitColours();
	GrSetDrawingRegion(0,0,MAP_WIDTH-1,MAP_HEIGHT-1);
	GrSetTextRegion(0,MAP_HEIGHT,MAP_WIDTH-1,SCREEN_HEIGHT-1);

	if (InitTiles() < 0) Die("Error Loading tiles");
	InitPlates();
	return 0;
	}
	
int
ShutdownGraphics()
	{
	GrShutdown();
	}

int
ClearMap()
	{
	int i,j,x,y,w,h,x1,y1,x2,y2;
	GrObject *gc;

	gc = GetTile(T_BLANK,SYSTEM_TILE,&x1,&y1,&x2,&y2);
	for(y=j=0; j<MAP_CELLS_DOWN; ++j, y+=CELL_SIZE)
	   for(x=i=0; i<MAP_CELLS_ACROSS; ++i, x+=CELL_SIZE)
		GrDrawObjectSubSectionNC(gc,x,y,x1,y1,x2,y2);
	return 0;
	}

/*
** Check if the terrain type (t) is one which requires merging with the
** surrounding terrain (e.g. lake shore) and if so, locate a piece of
** nearby land and return its terrain type.
*/

int
BaseTileForTerrain(int t, int cx, int cy)
	{
	TerrainSelector s;

	switch(t)
	   {
	   case T_SHALLOW_WATER:
		ClearTerrainSelector(&s);
		SetTerrainSelector(&s,T_SHALLOW_WATER);
		SetTerrainSelector(&s,T_DEEP_WATER);
		SetTerrainSelector(&s,T_UNDEFINED);
		InvertTerrainSelector(&s);
		break;
	   default:
		return T_UNDEFINED; break;
	   }

	return LocateSurroundingTile(cx,cy,&s);
	}

static int
LocateSurroundingTile(int cx, int cy, TerrainSelector *s)
	{
	int dx[8] = {-1,-1,0,1,1,1,0,-1};
	int dy[8] = {0,-1,-1,-1,0,1,1,1};
	unsigned int i,t;
	
	for(i=0; i<8; ++i)
	   {
	   t = Get_TerrainType(cy+dy[i],cx+dx[i]);
	   if (TestTerrainSelector(s,t)) return t;
	   }
	return T_UNDEFINED;
	}

/*
** NumVarTiles(T) returns the number of variation tiles available for base
** tile type T
*/
int
NumVarTiles(int t)
	{
	char fname[64];

	if (VarTiles[t] == NULL)
	   {
	   sprintf(fname,"images/%03d.raw",t);
	   VarTiles[t] = LoadTileSet(fname);
	   }

	if (VarTiles[t] != NULL)
	   return VarTiles[t]->Width / CELL_SIZE;
	else return 0;
	}

/*
*****************************************************************************
*/

static int
InitTiles()
	{
	int i;

	SystemTiles=NULL;
	for(i=0; i<256; ++i) VarTiles[i]=NULL;

	SystemTiles = LoadTileSet("images/system.raw");
	if (SystemTiles == NULL) Die("Cannot load System Tiles");
	return 0;
	}

/*
** Return a single tile from the system tile set which gives the images
** for the requested terrain type.
** Each tile is 16x16 pixels.
*/
GrObject *
GetTile(int which, int var, int *x1, int *y1, int *x2, int *y2)
	{
	if (var == SYSTEM_TILE)
	   {
	   *x1 = which * CELL_SIZE;
	   *y1 = 0;
	   *x2 = *x1 + CELL_SIZE - 1;
	   *y2 = *y1 + CELL_SIZE - 1;

	   if (*x1 >= SystemTiles->Width || 
	       *x2 >= SystemTiles->Width || 
	       *y1 >= SystemTiles->Height || 
	       *y2 >= SystemTiles->Height)
		Die("Request for nonexistant tile type=%d var=%d\n",which,var);
	   return SystemTiles;
	   }

	/*
	** Otherwise, we are being asked for a variant tile from one
	** of the other tile sets.
	*/
	if (VarTiles[which] == NULL)
	   {
	   char fname[64];

	   sprintf(fname,"images/%03d.raw",which);
	   VarTiles[which] = LoadTileSet(fname);
	   if (VarTiles[which] == NULL) Die("Cannot load Tileset %s\n",which);
	   }

	*x1 = var * CELL_SIZE;
	*y1 = 0;
	*x2 = *x1 + CELL_SIZE-1;
	*y2 = *y1 + CELL_SIZE-1;
	if (*x1 >= VarTiles[which]->Width || 
	    *x2 >= VarTiles[which]->Width || 
	    *y1 >= VarTiles[which]->Height || 
	    *y2 >= VarTiles[which]->Height)
		Die("Request for nonexistant tile type=%d var=%d\n",which,var);

	return VarTiles[which];
	}

static GrObject *
LoadTileSet(char *file)
	{
	FILE *z;
	char *dst,*src,fname[64],*data=NULL;
	char block[4096];
	GrObject *gc;
	int x,y,n,len;

	z = fopen(file,"rb");
	if (z==NULL) Die("LoadTileSet: Invalid file %s",file);

	/* Load the contents into memory */
	len=0; while((n = fread(block,1,4096,z)) > 0)
	   {
	   if (data==NULL) data=malloc(n);
	   else data = realloc(data,len+n);

	   if (data==NULL) Die("LoadTileSet: Malloc failed");
	   dst = &data[len]; src=block; len += n;
	   while(n--) *(dst++) = *(src++);
	   }
	fclose(z);

	/* Use the File size to calculate the number of pixels loaded
	** and allocate a linear context to hold these tiles
	*/
	gc = GrCreateObject(len/(CELL_SIZE * PIXEL_BYTES), CELL_SIZE);
	if (gc == NULL) Die("LoadTileSet: Object creation failed (Loaded %d bytes)",len);

	/* Copy the data across */
	src = data; dst = (char *)gc->data;
	while(len--) *(dst++) = *(src++);
	free(data);
	return gc;
	}

int GetColour(int cname)
	{
	return Colour[cname];
	}

#define SETCOLOUR(n,R,G,B) Colour[n] = (ushort)(((R)<<11) | ((G)<<6) | (B))

static int 
InitColours()
	{
	SETCOLOUR(BLACK,	0,0,0);
	SETCOLOUR(GREY,		15,15,15);
	SETCOLOUR(DARK_GREY,	8,8,8);
	SETCOLOUR(LIGHT_GREY,	22,22,22);
	SETCOLOUR(WHITE,	31,31,31);
	SETCOLOUR(RED,		31,0,0);
	SETCOLOUR(GREEN,	0,31,0);
	SETCOLOUR(BLUE,		0,0,31);
	SETCOLOUR(DARK_RED,	15,0,0);
	SETCOLOUR(DARK_GREEN,	0,15,0);
	SETCOLOUR(DARK_BLUE,	0,0,15);
	SETCOLOUR(YELLOW,	25,25,0);
	SETCOLOUR(PINK,		25,15,15);
	SETCOLOUR(LIGHT_GREEN,	15,31,15);
	SETCOLOUR(LIGHT_BLUE,	15,15,31);
	SETCOLOUR(ORANGE,	28,0,0);
	SETCOLOUR(BROWN,	0,28,28);
	return 0;
	}

/*
** ScreenText(str,x,y) : Displays the string "str" on the screen at coords
** 		      (x,y)
*/
int
ScreenText(char *buf, int x, int y)
	{
	}

/************************************************************************/

/*
** Return a random variant for the requested system tile, or 0 if there
** are no variants available.
*/

int 
RandomTileVariant(int t)
	{
	int n = NumVarTiles(t);

	if (n==0) return 0;
	return Random(n);
	}
