/*
** Symbol table maintenance routines
*/

#include "parse.h"

#define MAX_SYMS	1024
#define MAX_TABLES	256

typedef struct
	{
	int nsyms;
	char *Sym[MAX_SYMS];
	} SymList;

SymList Tables[MAX_SYMS];
int tablemap[MAX_TABLES];

int
NewTable()
	{
	int i,n;
	SymList *s;

	for(n=0; n<MAX_TABLES; ++n) if (!tablemap[n])
		{
		tablemap[n]=1;

		s = &Tables[n];
		for(i=0; i<MAX_SYMS; ++i) s->Sym[i]=NULL;
		s->nsyms=0;
		return n;
		}
	return -1;
	}

char *
FindSymbol(int tn, char *str)
	{
	SymList *s;
	int i,len;

	s = &Tables[tn];
	len = strlen(str);
	for(i=0; i<s->nsyms; ++i)
	   if (!strncmp(s->Sym[i],str,len) && s->Sym[i][len] == ':')
		return s->Sym[i];

	return NULL;
	}

int
FreeTable(int n)
	{
	int i;
	SymList *s;

	if (!tablemap[n]) return(0);

	tablemap[n]=0;
	s = &Tables[n];
	for(i=0; i<s->nsyms; ++i) if (s->Sym[i])
		free(s->Sym[i]);
	s->nsyms=0;
	return 0;
	}

int
PushSymbol(int n, char *str,...)
	{
	va_list arg;
	char buf[4096];
	int i;
	SymList *s;

	s = &Tables[n];
	if (s->nsyms==MAX_SYMS) return 1;

	va_start(arg,str);
	vsprintf(buf,str,arg);

	i=s->nsyms++;
	s->Sym[i] = (char *)malloc(strlen(buf)+1);
	strcpy(s->Sym[i],buf);
	va_end(arg);
	return 0;
	}

void
ReplaceSymbol(int tn, int n, char *newstr)
	{
	SymList *s;

	if (!tablemap[tn]) Fail("Access to unmapped table");
	s = &Tables[tn];
	if (n>= s->nsyms) Fail("Access beyond end of table");

	s->Sym[n] = (char *)realloc(s->Sym[n],strlen(newstr)+1);
	if (s->Sym[n] == NULL) Fail("Out of memory");

	strcpy(s->Sym[n],newstr);
	}

/*
** PopSymbols() removes the last N entries from the table, shortening it.
*/
int
PopSymbols(int tn, int num)
	{
	int i,n;
	SymList *s;

	s = &Tables[tn];
	n = s->nsyms;

	if (num>n) 
		{
		fprintf(stderr,"PopSymbols: tried to remove %d symbols when there are only %d\n",num,n);
		return 1;
		}

	for(i = n-num; i < n; ++i)
	   free(s->Sym[i]);

	s->nsyms -= num;
	return 0;
	}

void
GetReturnVal(int tn, char *func, char *rval)
	{
	if (GetSymData(tn,func,1,rval))
	   Fail("Internal symbol format error");
	}
	
void
DumpTable(int tn)
	{
	SymList *s;
	int i;

	if (tn==v_table) printf("v_table:");
	else if (tn==gv_table) printf("gv_table:");
	else if (tn==p_table) printf("p_table:");
	else if (tn==t_table) printf("t_table:");
	else if (tn==gt_table) printf("gt_table:");
	else printf("unknown table:");
	printf("SP=%d:",StackPointer);
	s = &Tables[tn];
	printf(" (%d symbols)\n",s->nsyms);
	for(i=0; i<s->nsyms; ++i)
	   printf("\t%s\n",s->Sym[i]);
	printf("\n");
	}

void
DumpTables()
	{
	int i;

	for(i=0; i<MAX_TABLES; ++i) if (tablemap[i])
	   {
	   printf("Dump of symbol table %d:\n",i);
	   DumpTable(i);
	   }
	}

int
GetSymData(int tn, char *symname, int field, char *rval)
	{
	char *str;
	int capture,i,j;

	str = FindSymbol(tn,symname);
	if (str==NULL) return 1;

	GetField(str,field,rval);
	return 0;
	}

int
CopySymbols(int src_t, int dst_t, int check)
	{
	SymList *src,*dst;
	char name[32],type[32];
	int nelem,n,i,j;

	if (src_t == dst_t) Fail("CopySymbols: dest and src are same table");
	src = &Tables[src_t];
	dst = &Tables[dst_t];

	j=dst->nsyms;
	for(i=0; i<src->nsyms; ++i,++j)
	   {
	   if (check)
	   	{
	   	GetVariable(src_t,i,name,type,&nelem);
	   	if (FindSymbol(dst_t,name)) Fail("Duplicate symbol \"%s\" found",name);
	   	}

	   dst->Sym[j] = (char *)malloc(strlen(src->Sym[i])+1);
	   strcpy(dst->Sym[j],src->Sym[i]);
	   ++dst->nsyms;
	   }
	return 0;
	}

static int pos=0;
static int istack[1024];

void
PushInt(int val)
	{
	if (pos==1023) Fail("Internal stack overflow");
	istack[pos++] = val;
	}

int
PopInt()
	{
	if (pos==0) Fail("Pop() on empty stack");
	return istack[--pos];
	}

static char strbuf[2048];
static int strpos=0;

void
PushString(char *s)
	{
	PushInt(strpos);
	strcat(strbuf,s);
	strpos += strlen(s);
	}

void
PopString(char *s)
	{
	strpos = PopInt();
	strcpy(s,&strbuf[strpos]);
	strbuf[strpos]=0;
	}

int
NumSymbols(int tn)
	{
	return Tables[tn].nsyms;
	}

void
DeclareVariable(int tn, char *type, char *name,int ndims)
	{
	int i,n;
	SymList *s;
	char tmp[256];
	instruct *in;

	if (name && FindSymbol(tn,name))
	   Fail("Variable %s is already declared",name);

	if (name==NULL)
	   {
	   if (scope==LOCAL) Fail("Local variables must have TYPE and NAME");
	   name="<NONE>"; /* allowed in prototypes */
	   }

	if (scope==LOCAL || scope==MAYBE_LOCAL)
	   {
	   if (ndims==0)
		{
		in = BuildInstruction(OP_PUSH,1,"i%s",type);
		PushInstruction(in);
	   	sprintf(tmp,"%s:%s:%d:s%d",name,type,ndims,StackPointer);
	   	StackPointer += GetTypeSize(type);
		}
	   else
		{
		strcpy(tmp,"@"); for(i=0; i<ndims; ++i) strcat(tmp," @");
		in = BuildInstruction(OP_ADECL,ndims+2,"t%s %s",type,tmp);
		PushInstruction(in);
	   	sprintf(tmp,"%s:%s:%d:s%d",name,type,ndims,StackPointer);
		StackPointer += GetTypeSize("int");
		}
		
	   }
	else
	   {
	   if (ndims==0)
		{
		/* global symbol, we can allocate it now */
	   	sprintf(tmp,"%s:%s:%d:h%d",name,type,ndims,HeapOffset);
		HeapOffset += GetTypeSize(type);
		}
	   else
		{
		strcpy(tmp,"@"); for(i=1; i<ndims; ++i) strcat(tmp," @");
		in = BuildInstruction(OP_ADECL,ndims+2,"t%s g%s %s",type,name,tmp);
		PushInstruction(in);
		sprintf(tmp,"%s:%s:%d:h_",name,type,ndims);
		}
	   }
	PushSymbol(tn,tmp);
	}

int
GetVariable(int tn, int n, char *name, char *type, int *nelem)
	{
	char *data;
	int i,j;

	if (n<0 || n>Tables[tn].nsyms)
	   Fail("Symbol table access error");

	data = Tables[tn].Sym[n];
	for(i=j=0; data[i] && data[i] != ':'; ++i) name[i]=data[i];
	name[i]=0;

	if (data[i]!=':') Fail("Symbol table format error");
	++i;
	while(data[i] && data[i]!=':') type[j++] = data[i++];
	type[j]=0;
	if (data[i]!=':') Fail("Symbol table format error");
	++i;

	*nelem = atoi(data+i);
	return 0;
	}

int
GetField(char *str, int field, char *rval)
	{
	int i,j,capture;

	if (field==0) capture=1; else capture=0;
	for(i=j=0; str[i]; ++i)
	   switch(str[i])
		{
		case ':': --field; if (field==0) capture=1;
			  else if (field<0) 
				{
				rval[j]=0;
				return 0;
				}
			 break;
		default: if (capture) rval[j++] = str[i];
			break;
		}

	rval[j]=0;
	return 0;
	}

/*
** funcname[] contains the name of this function, and the table
** fh_table contains all the parameters.
**
** functype[] contains the return type string.
*/
int
CheckParameters(char *funcname, int pt)
	{
	int i,dims,n,size=0;
	char *data,tmp[32],name[32],type[32];

	/* Locate the prototype entry */
	data = FindSymbol(p_table,funcname);
	if (data==NULL) Fail("Could not locate prototype for %s",funcname);

	/* Check that the number of parameters matches */
	GetField(data,2,tmp);
	n = NumSymbols(pt);
	if (atoi(tmp) != n) Fail("Incorrect number of parameters");

	/*
	** Check each parameter
	*/

	for(i=0; i<n; ++i)
	   {
	   GetVariable(pt,i,name,type,&dims);
	   GetField(data,3+3*i,tmp);
	   if (strcmp(tmp,type))
		Fail("Parameter %d has invalid type \"%s\"",i+1,type);
	   GetField(data,5+3*i,tmp);
	   if (atoi(tmp) != dims) 
		Fail("Parameter %d has incorrect number of dimensions",i+1);
	   if (dims==0) size += GetTypeSize(type);
	   else size+=4;
	   }

	/* Now adjust the address of the parameters so that they are
	** before the current stack pointer (since they are args)
	*/
	for(i=0; i<n; ++i)
	   SetVariableLocation(pt,i, GetVariableLocation(pt,i) - size);
	}

int
GetVariableLocation(int tn, int n)
	{
	char rval[32];
	SymList *s;

	s = &Tables[tn];
	if (!tablemap[tn]) Fail("Access to unmapped table");

	if (n>=s->nsyms) Fail("Access beyond end of symbol table");

	GetField(s->Sym[n],3,rval);
	return atoi(rval+1);
	}

void
SetVariableLocation(int tn, int n, int loc)
	{
	SymList *s;
	char name[32],type[32],tmp[128];
	int ndims;

	s = &Tables[tn];
	if (!tablemap[tn]) Fail("Access to unmapped table");

	if (n>=s->nsyms) Fail("Access beyond end of symbol table");

	/* Location is the 4th field: NAME TYPE DIMS LOCATION */
	GetVariable(tn,n,name,type,&ndims);

	sprintf(tmp,"%s:%s:%d:s%d",name,type,ndims,loc);
	ReplaceSymbol(tn,n,tmp);
	return;
	}


int
GetTypeSize(char *type)
	{
	int i,size=0,len,ndim;
	SymList *s;
	char *data,rval[32],ttype[32];

	if (sscanf(type,"%s %d",rval,&ndim) == 2)
	   {
	   if (ndim==0) return GetTypeSize(rval);
	   else return GetTypeSize("int");
	   }

	data = FindSymbol(t_table,type);
	if (data==NULL) {DumpTable(t_table); Fail("%s: Invalid type",type);}
	GetField(data,1,ttype);

	/* basic type - has size in 4th field */
	if (!strcmp(type,ttype))
	   {
	   GetField(data,3,rval);
	   return atoi(rval);
	   }

	/* simple typedef */
	GetField(data,2,rval);
	if (atoi(rval) == 0)
	   return GetTypeSize(ttype);

	/* compound type */
	for(i=atoi(rval)-1; i>=0; --i)
	   {
	   GetField(data,5 + 3*i,rval);
	   if (atoi(rval)>0) size += GetTypeSize("int");
	   else
		{
	   	GetField(data,3 + 3*i,rval);
	   	size += GetTypeSize(rval);
		}
	   }

	return size;
	}

void
WriteGlobals()
	{
	SymList *s;
	int i,n;

	n = NumSymbols(gv_table);
	if (n==0) return;

	s = &Tables[gv_table];
	printf(".global\n");
	for(i=0; i<n; ++i) printf("\t%s\n",s->Sym[i]);
	printf("\n");
	}

void
WriteTypes()
	{
	SymList *s;
	char *data,rval[32];
	int i,ii,nelem,n;

	s = &Tables[t_table];
	n = NumSymbols(t_table);
	printf(".types\n");
	for(i=0; i<n; ++i)
	   {
	   data = s->Sym[i];
	   GetField(data,0,rval);
	   printf("\t%s %d",rval,GetTypeSize(rval));
	   GetField(data,2,rval); nelem=atoi(rval);
	   for(ii=0; ii<nelem; ++ii)
		{
		GetField(data,3+3*ii,rval);
		printf(" %s",rval);
		GetField(data,5+3*ii,rval);
		printf(" %s",rval);
		}
	   putchar('\n');
	   }
	}

int
LocateStructElement(char *type, char *name, char *rtype, int *offset, int *size)
	{
	char *data,rval[32];
	int found,i,nelem,ndims;

	data = FindSymbol(t_table,type);
	if (data==NULL) Fail("Unknown type: %s",type);
	GetField(data,1,rval);
	if (strcmp(rval,"struct")) Fail("%s is not a struct",type);
	GetField(data,2,rval); nelem = atoi(rval);
	*offset=0; rtype[0]=0;
	*size=0;
	for(i=found=0; !found && i<nelem; ++i)
		{
		GetField(data,4+3*i,rval);
		if (!strcmp(rval,name))
		   {
		   GetField(data,3+3*i,rtype);
		   GetField(data,5+3*i,rval); ndims = atoi(rval);
		   if (ndims>0) {strcat(rtype," "); strcat(rtype,rval);}
		   *size = GetTypeSize(rtype);
		   found=1;
	  	   }
		else
		   {
		   GetField(data,3+3*i,rval);
		   *offset += GetTypeSize(rval);
		   }
		}
	return found;
	}
