/*
** This is the medium-level graphics subsystem. Here we are dealing with
** a collection of "plates" which build up the overall screen.
**
** Each plate is an NxN array of tiles, and can float around the screen
** as required by panning/scrolling requests.
**
*/

#include "game.h"

static struct plate
	{
	int x,y;		/* top left screen coords of plate */
	int px,py;		/* Top left map coords of plate */
	int OnScreen;		/* 1 if this plate is visible */
	int dirty;
	GrObject *data;
	} PlateList[NPLATES];

static short PlateMap[MAP_CELLS_ACROSS * MAP_CELLS_DOWN];

static void LoadPlate(struct plate *, int px, int py);
static void LoadPlateSubSection(GrObject *data, int dx, int dy, int cx, int cy, int w, int h);
static void DisplayPlate(int);
static void RedrawRHS(void);
static void RedrawLHS(void);
static void RedrawBottom(void);
static void RedrawTop(void);
static struct plate *GetPlate(int,int, int *, int *);
static void Dump_Plates(void);

static int PX=0;	/* X Screen coord plate boundary offset */
static int PY=0;	/* Y Screen coord plate boundary offset */

/*
** This function maps a screen coordinate (x,y) onto a plate and internal
** plate coords (px,py).
*/

static struct plate *
GetPlate(int x,int y, int *px, int *py)
	{
	int i,j,x1,y1;
	static int last_plate=-1;

	/* See if there is a plate already available for this cell */
	if (last_plate >=0) i=last_plate; else i=0;

	for(j=0; j<NPLATES; ++j,i= (i<NPLATES-1) ? i+1: 0)
	   if (PlateList[i].OnScreen &&
		x >= PlateList[i].x &&
		y >= PlateList[i].y &&
		x - PlateList[i].x < PLATEPIXWIDTH &&
		y - PlateList[i].y < PLATEPIXHEIGHT)
	   {
	   *px = x - PlateList[i].x;
	   *py = y - PlateList[i].y;
	   last_plate=i;
	   return & (PlateList[i]);
	   }

	/* Locate a free (offscreen) plate and reassign it */
	for(i=0; i<NPLATES; ++i) if (!PlateList[i].OnScreen)
	   {
	   x1 = x-PX; if (x1<0) x1+=PLATEPIXWIDTH;
	   x1 = x1 % PLATEPIXWIDTH;
	   y1 = y-PY; if (y1<0) y1+=PLATEPIXHEIGHT;
	   y1 = y1 % PLATEPIXHEIGHT;
	   PlateList[i].x = x - x1;
	   PlateList[i].y = y - y1;
	   *px=x1; *py=y1;
	   last_plate=i;
	   return &(PlateList[i]);
	   }

	GrShutdown();
	printf("Cannot allocate a plate for (%d,%d)\n",x,y);
	Dump_Plates();
	exit(0);
	}

void
InitPlates()
	{
	int n;

	for(n=0; n<NPLATES; ++n)
	   {
	   PlateList[n].data = GrCreateObject(PLATEPIXWIDTH,PLATEPIXHEIGHT);
	   if (PlateList[n].data == NULL) Die("Malloc Error in InitPlates");
	   PlateList[n].x=0; PlateList[n].y=0; PlateList[n].OnScreen=0;
	   PlateList[n].dirty=0;
	   }
	}

/*
** Display Plate n. Assume that it is visible
*/
static void
DisplayPlate(int n)
	{
	struct plate *pl = &PlateList[n];
	int x,y;
	int x1,y1,x2,y2;

 	x=pl->x; y=pl->y;

	/* Optimisation: maybe we don't need to clip */
	if (x>=0 && y>=0 && x<(MAP_WIDTH - PLATEPIXWIDTH) &&
	   y<(MAP_HEIGHT - PLATEPIXHEIGHT))
	     GrDrawObjectNC(pl->data,x,y);
	else GrDrawObject(pl->data,x,y);
	}

int
MapRefresh()
	{
	/*
	** Locate all plates which are at least partially visible and redraw
	** them onto the screen.
	*/
	int n;

#ifdef USE_JUGL
	GrNextFrame();
#endif
	for(n=0; n<NPLATES; ++n) if (PlateList[n].OnScreen)
	    if (1 || PlateList[n].dirty)
		{DisplayPlate(n); PlateList[n].dirty=0;}

	RedrawRegions();
	RefreshText();

#ifdef USE_JUGL
	GrShowFrame();
#endif
	return 0;
	}

/*
** Draw Cell (cy,cx) onto internal screen at screen coordinates (Y,X)
*/
void
DrawCell(int cy, int cx, int Y, int X)
	{
	unsigned int px,py;
	struct plate *pl;

	/* Locate a plate and coords for this cell */
	pl = GetPlate(X,Y,&px,&py);
	if (pl == NULL) Die("Cannot locate valid plate");

	if (pl->OnScreen) return;	/* this cell is already here */

	/*
	** Calculate the cell coords(cx,cy) for the cell at the top-left
	** corner of this plate. We can do this from the pixel coords (px,py)
	** which are internal to the plate.
	*/
	cy = cy-py/CELL_SIZE; cx = cx-px/CELL_SIZE;
	LoadPlate(pl,cx,cy);

	/*
	** Now check the animation pool to see whether there are any frames
	** to be drawn onto this plate. Convert all coords to pixel coords.
	*/
	if (pl->OnScreen == 0)  /* Newly allocated plate */
	   {
	   cx*=CELL_SIZE; cy*=CELL_SIZE;
	   pl->OnScreen=1;
	   UpdateEntities(cx,cy,cx+PLATEPIXWIDTH-1,cy+PLATEPIXHEIGHT-1);
	   }
	pl->dirty=1;
	}

/***************************************************************************
** LoadPlate: reload plate n from the terrain. Top left cell is (sx,sy)
**
** This function makes some assumtions:
**	- Plates are always aligned on cell boundaries
**
****************************************************************************/
static void
LoadPlate(struct plate *pl, int sx, int sy)
	{
	/* Top Left map coords for plate */
	pl->px = sx * CELL_SIZE;
	pl->py = sy * CELL_SIZE;

	LoadPlateSubSection(pl->data,0,0,sx,sy,PLATECELLWIDTH,PLATECELLHEIGHT);
	}

/*
** Load a section of a plate from the terrain map. Arguments are:
** 	data		The destination GrObject
**	(dx,dy)		Top left destination coords in data
**	(cx,cy)		Top left terrain coords
**	(w,h)		Width and height in cells to load
*/
static void
LoadPlateSubSection(GrObject *data, int dx, int dy, int sx, int sy, int w, int h)
	{
	int i,j,x1,y1,x2,y2,cx,cy,d1;
	int ter,ter1,whichtile;
	GrObject *gc;

	d1=dx;
	for(j=0,cy=sy; j<h; ++j,++cy,dy+=CELL_SIZE)
	  for(i=0,cx=sx,dx=d1; i<w; ++i,++cx,dx+=CELL_SIZE)
		{
		ter = Get_TerrainType(cy,cx);
		whichtile = Get_TerrainImageIndex(cy,cx);
		ter1 = BaseTileForTerrain(ter,cx,cy);

		if (ter1 != T_UNDEFINED)
	   	   {
	   	   gc = GetTile(ter1,SYSTEM_TILE,&x1,&y1,&x2,&y2);
		   GrMergeObjectSubSectionNC(data,gc,dx,dy,x1,y1,x2,y2,GrWRITE);
	   	   gc = GetTile(ter,whichtile,&x1,&y1,&x2,&y2);
		   GrMergeObjectSubSectionNC(data,gc,dx,dy,x1,y1,x2,y2,GrIMAGE);
	   	   }
		else
	   	   {
	   	   gc = GetTile(ter,whichtile,&x1,&y1,&x2,&y2);
		   GrMergeObjectSubSectionNC(data,gc,dx,dy,x1,y1,x2,y2,GrWRITE);
	   	   }
		}

	}

/*
** Reload the given region of the map from the terrain cache into the
** plates. (x1,y1) is top left, (x2,y2) is bottom right.
*/

void
RedrawMapSubSection(int x1, int y1, int x2, int y2)
	{
	int i,area;
	int cx1,cy1,cx2,cy2,px1,py1,px2,py2;

	area = (x2-x1+1) * (y2-y1+1);
	if (area<=0) return;

	for(i=0; i<NPLATES; ++i) if (PlateList[i].OnScreen)
	   {
	   px1 = PlateList[i].px; px2 = px1+PLATEPIXWIDTH-1;
	   py1 = PlateList[i].py; py2 = py1+PLATEPIXHEIGHT-1;
	   if (Overlap_c(px1,py1,px2,py2,x1,y1,x2,y2,&cx1,&cy1,&cx2,&cy2))
		{
		/*
		** (cx1,cy1) - (cx2,cy2) is a rectangle to be redrawn in 
		** map pixel coords. Convert it to cell boundaries.
		*/
		LoadPlateSubSection(PlateList[i].data,cx1-px1,cy1-py1,
			cx1/CELL_SIZE, cy1/CELL_SIZE,
			(cx2-cx1+1)/CELL_SIZE, (cy2-cy1+1)/CELL_SIZE);
		area -= (cy2-cy1+1) * (cx2-cx1+1);
		if (area<=0) return;
		}
	   }
	}

void
MapScroll(int dx, int dy)
	{
	int i,j,mx,my,rh,rv;
	int omx,omy;

	if (dx==0 && dy==0) return;

	/*
	** We cannot scroll by more than a cell width at a time, lest we
	** stuff up our optimised redraw-just-the leading-edge code
	** so fake it with multiple calls.
	*/
	while (dx > 12) {MapScroll(12,0); dx -= 12;}
	while (dx < -12) {MapScroll(-12,0); dx += 12;}
	while (dy > 12) {MapScroll(0,12); dy -= 12;}
	while (dy < -12) {MapScroll(0,-12); dy += 12;}

	for(i=0; i<NPLATES; ++i) if (PlateList[i].OnScreen)
	   {
	   PlateList[i].x -= dx;
	   PlateList[i].y -= dy;
	   PlateList[i].dirty = 1;  /* force a redraw */
	   if (PlateList[i].x<= -PLATEPIXWIDTH || PlateList[i].y<= -PLATEPIXHEIGHT ||
		PlateList[i].x>=MAP_WIDTH || PlateList[i].y>=MAP_HEIGHT)
		PlateList[i].OnScreen=0;
	   }
	GetMapCentre(&omx,&omy);
	ScrollMapCentre(dx,dy);
	GetMapCentre(&mx,&my);

	PX-=dx; PY-=dy; i=PLATEPIXWIDTH;
	if (PX>=i) PX-=i; if (PX<=-i) PX+=i;
	if (PY>=PLATEPIXHEIGHT) PY-=PLATEPIXHEIGHT;
	if (PY<=-PLATEPIXHEIGHT) PY+=PLATEPIXHEIGHT;

	/* Has a new cell become visible on RHS?  */
	if (dx>0) {i=mx+MAP_WIDTH/2-1; j=omx+MAP_WIDTH/2-1;
	   if (i/CELL_SIZE != j/CELL_SIZE) RedrawRHS();
		}
	if (dx<0) {i=mx-MAP_WIDTH/2; j=omx-MAP_WIDTH/2;
	   if (i/CELL_SIZE != j/CELL_SIZE) RedrawLHS();
		}
	if (dy>0) {i=my+MAP_HEIGHT/2-1; j=omy+MAP_HEIGHT/2-1;
	   if (i/CELL_SIZE != j/CELL_SIZE) RedrawBottom();
		}
	if (dy<0) {i=my-MAP_HEIGHT/2; j=omy-MAP_HEIGHT/2;
	   if (i/CELL_SIZE != j/CELL_SIZE) RedrawTop();
		}
	}

static void
RedrawRHS()
	{
	int i,x,y,mx,my;
	int cx,cy;

	/* redraw the cells down the RHS column */
	/* calculate (cx,cy) as the cell coord for top of RHS column */
	GetMapCentre(&mx,&my);
	cx = (mx + MAP_WIDTH/2) / CELL_SIZE;
	cy = (my - MAP_HEIGHT/2) / CELL_SIZE;

	ScreenCoords(cx,cy,&x,&y);

	for(i=0; i<MAP_CELLS_DOWN + 1; ++i, ++cy, y += CELL_SIZE)
	   DrawCell(cy,cx,y,x);
	}

static void
RedrawLHS()
	{
	int i,x,y,mx,my;
	int cx,cy;

	/* redraw the cells down the RHS column */
	/* calculate (cx,cy) as the cell coord for top of RHS column */
	GetMapCentre(&mx,&my);
	cx = (mx - MAP_WIDTH/2) / CELL_SIZE;
	cy = (my - MAP_HEIGHT/2) / CELL_SIZE;

	ScreenCoords(cx,cy,&x,&y);

	for(i=0; i<MAP_CELLS_DOWN + 1; ++i, ++cy, y += CELL_SIZE)
	   DrawCell(cy,cx,y,x);
	}

static void
RedrawBottom()
	{
	int i,x,y,mx,my;
	int cx,cy;

	/* redraw the cells down the RHS column */
	/* calculate (cx,cy) as the cell coord for top of RHS column */
	GetMapCentre(&mx,&my);
	cx = (mx - MAP_WIDTH/2) / CELL_SIZE;
	cy = (my + MAP_HEIGHT/2) / CELL_SIZE;

	ScreenCoords(cx,cy,&x,&y);

	for(i=0; i<MAP_CELLS_ACROSS + 1; ++i, ++cx, x += CELL_SIZE)
	   DrawCell(cy,cx,y,x);
	}

static void
RedrawTop()
	{
	int i,x,y,mx,my;
	int cx,cy;

	/* redraw the cells down the RHS column */
	/* calculate (cx,cy) as the cell coord for top of RHS column */
	GetMapCentre(&mx,&my);
	cx = (mx - MAP_WIDTH/2) / CELL_SIZE;
	cy = (my - MAP_HEIGHT/2) / CELL_SIZE;

	ScreenCoords(cx,cy,&x,&y);

	for(i=0; i<MAP_CELLS_ACROSS + 1; ++i, ++cx, x += CELL_SIZE)
	   DrawCell(cy,cx,y,x);
	}

/*
** Call this after manually moving the screen coords (MX,MY)
*/
void
InvalidateALLPlates()
	{
	int i;
	int mx,my;

	GetMapCentre(&mx,&my);

	for(i=0; i<NPLATES; ++i)
	   {PlateList[i].OnScreen=0; PlateList[i].dirty=1;}
	PX = (mx-MAP_WIDTH/2) % PLATEPIXWIDTH;
	PY = (my-MAP_HEIGHT/2) % PLATEPIXHEIGHT;
	}

/*
** Invalidate all plates which overlap with the passed rectangle
*/

void
InvalidatePlates(int ax1, int ay1, int ax2, int ay2)
	{
	int px1,py1,px2,py2;
	int i;

	for(i=0; i<NPLATES; ++i) if (PlateList[i].OnScreen)
	   {
	   px1 = PlateList[i].px; py1 = PlateList[i].py;
	   px2 = px1 + PLATEPIXWIDTH-1; py2 = py1 + PLATEPIXHEIGHT-1;
	   if (Overlap_p(ax1,ay1,ax2,ay2,px1,py1,px2,py2))
		LoadPlate(&PlateList[i], PlateList[i].px / CELL_SIZE,
					 PlateList[i].py / CELL_SIZE);
	   }
	}

/*
** This is the other interface to the mid-level graphics subsystem, allowing
** an arbitrary image to be drawn onto the plates which make up the displayed
** screen. The coords supplied here are screen coordinates, and this routine
** locates the appropriate plates and draws the pieces of the supplied image
** such that a MapRefresh() will display it in the appropriate place.
**
** (dx,dy) are in absolute pixel coordinates (0-64k)
*/
void
DrawImage(GrObject *gc, int dx, int dy, int sx1,int sy1,int sx2, int sy2, int op)
	{
	int i,ii,j=0,area,w,h;
	int mx,my;
	int dx1,dy1,dx2,dy2,px1,px2,py1,py2,cx1,cx2,cy1,cy2;

	w = (sx2-sx1+1); h=(sy2-sy1+1);
	area = w*h;

	dx1=dx; dy1=dy; dx2=dx1+w-1; dy2=dy1+h-1;

	/* Locate the plates which overlap with the desired region */
	for(i=0; i<NPLATES; ++i) if (PlateList[i].OnScreen)
	   {
	   px1=PlateList[i].px; py1=PlateList[i].py;
	   px2=px1+PLATEPIXWIDTH-1; py2=py1+PLATEPIXHEIGHT-1;

	   /* Check for overlap, and clip into rectangle c */
	   if (!Overlap_c(dx1,dy1,dx2,dy2,px1,py1,px2,py2,
			&cx1,&cy1,&cx2,&cy2)) continue;

	   /* Write out this image subsection */
	   mx = sx1 + (cx1-dx1);
	   my = sy1 + (cy1-dy1);

	   GrMergeObjectSubSectionNC(PlateList[i].data,gc,cx1-px1,cy1-py1,
	   	mx,my,
		mx + (cx2-cx1),
		my + (cy2-cy1), op);
	   PlateList[i].dirty=1;

	   area -= (cy2-cy1) * (cx2-cx1);
	   if (area<0) Die("DrawImage: wrote too many pixels");
	   if (area==0) break;
	   }
	}

/*
** Grab a rectangular region off the screen (indirectly - through the plates)
** and store it into dst
*/

void
GetMapSubSection(GrObject *dst, int dx, int dy, int ax1,int ay1, int ax2, int ay2)
	{
	int i,area;
	int cx1,cy1,cx2,cy2,px1,py1,px2,py2;

	area = (ax2-ax1+1) * (ay2-ay1+1);

	/* Locate the plates which overlap with the desired region */
	for(i=0; i<NPLATES; ++i) if (PlateList[i].OnScreen)
	   {
	   px1=PlateList[i].px; py1=PlateList[i].py;
	   px2=px1+PLATEPIXWIDTH-1; py2=py1+PLATEPIXHEIGHT-1;

	   /* Test for overlap and clip */
	   if (Overlap_c(ax1,ay1,ax2,ay2,px1,py1,px2,py2,&cx1,&cy1,&cx2,&cy2))
		{
	   	/* Write out this image subsection */
	   	GrMergeObjectSubSectionNC(dst,PlateList[i].data,
		    dx + (cx1-ax1),dy + (cy1-ay1),
	   		cx1 - px1,
	   		cy1 - py1,
			(cx1-px1) + (cx2-cx1),
			(cy1-py1) + (cy2-cy1),
			GrWRITE);

	   	area -= (cy2-cy1) * (cx2-cx1);
	   	if (area<0) Die("DrawImage: wrote too many pixels");
	   	if (area==0) return;
		}
	   }
	}

static void
Dump_Plates()
	{
	int i;
	struct plate *p;

	for(i=0; i<NPLATES; ++i)
	   {
	   printf("Plate %d: (%d,%d) OnScreen=%d dirty=%d\n",i,
		PlateList[i].x,PlateList[i].y,
		PlateList[i].OnScreen, PlateList[i].dirty);
	   Log("Plate %d: (%d,%d) OnScreen=%d dirty=%d\n",i,
		PlateList[i].x,PlateList[i].y,
		PlateList[i].OnScreen, PlateList[i].dirty);
	   }

	}
