/*
** This modules handles image animation
*/

#include "game.h"

#define SPRITES_PER_PLANE	8192
#define MAX_SPRITES		32768

#define STRTOK_TOKSTR		" \n\r\t"
/*
** Status values for Entities
*/
#define RUNNING		1
#define STOPPED		2
#define DELETED		3

/* Generic struct to hold one sprite */
typedef struct
	{
	char *Name;			/* Sprite name */
	int Width,Height;		/* Dimensions of sprite */
	GrObject **gc;			/* Array of images */
	int nFrames;			/* Number of frames in this sprite */
	} Sprite;

/* An Active sprite (instantiation of Sprite) */
typedef struct
	{
	Sprite *data;			/* Pointer to sprite */
	int x,y;			/* On-Screen location */
	int cFrame;			/* currently displayed frame */
	unsigned int status;
	} Entity;

typedef struct
	{
	Entity *EList[SPRITES_PER_PLANE];
	int count;				/* Number of Entities in plane */
	} planetype;

static planetype Plane[NPLANES];

static Sprite *SList[MAX_SPRITES];
static int nSprites = 0;
static char *SPRITEDIR = "sprites";

static char *__Current_Sprite;
static int __Current_Line;
static char * RequiredToken(void);
static int RequiredNumericToken(void);
static void InitSprites(void);
static GrObject *LoadFrame(char *fname, Sprite *sp);

static int __done_init=0;

/*
** Return an identifier for sprite with a particular name
*/
int
GetSpriteID(char *sname)
	{
	int i;

	for(i=0; i<nSprites; ++i) if (!strcmp(sname,SList[i]->Name)) return i;
	return -1;
	}

/*
** GetSpriteData returns a (short *) pointing to the data area for the
** sprite.
*/
unsigned short *
GetSpriteData(int sid, int fr)
	{
	if (sid<0 || sid>= nSprites) return NULL;
	if (fr<0 || fr >= SList[sid]->nFrames) return NULL;

	return (unsigned short *) SList[sid]->gc[fr]->data;
	}

/*
** Returns the width and height for the requested sprite
*/
void
GetSpriteWH(int sid, int *w, int *h)
	{
	if (sid<0 || sid>= nSprites) Die("GetSpriteWH: Invalid SID");

	*w = SList[sid]->Width;
	*h = SList[sid]->Height;
	}

void
SetSpritePixel(int sid, int fr, int x, int y, int val)
	{
	unsigned short *addr;
	if (sid<0 || sid>= nSprites) Die("GetSpriteWH: Invalid SID");
	if (fr<0 || fr >= SList[sid]->nFrames) Die("GetSpriteID: Invalid frame");

	addr = (unsigned short *) SList[sid]->gc[fr]->data;
	addr[SList[sid]->Width * y + x] = val;
	}

/*
** Loads the sprites contained in the file fname. returns the number of
** sprites successfully loaded.
*/
int
LoadSprites(char *name)
	{
	int i,framenum=0,insprite=0,SpritesLoaded=0;
	char fname[128],*sname;
	char line[256],*ch;
	Sprite *sp=NULL;
	FILE *z;

	if (!__done_init) InitSprites();

	sprintf(fname,"%s/%s",SPRITEDIR,name);
	z = fopen(fname,"rb"); if (z==NULL) return -2;

	__Current_Line=0;
	__Current_Sprite = fname;
	while(1)
	   {
	   __Current_Line++;
	   line[0] = 0; fgets(line,127,z); if (feof(z)) break;
	   ch = strtok(line,STRTOK_TOKSTR); if (ch==NULL) continue;
	   if (ch[0]=='#') continue;

	   if (!_my_strcasecmp(ch,"sprite"))
		{
		if (insprite) Die("LoadSprites: Line %d: Already inside a sprite ",__Current_Line);
		insprite=1;
		sname = RequiredToken();
		/* Do we already have it ? (*/
		for(i=0; i<nSprites; ++i) if (!strcmp(sname,SList[i]->Name))
		   Die("LoadSprites (%s): Already have sprite %s",fname,sname);

		/* Allocate memory for the sprite */
		SList[nSprites] = (Sprite *) malloc(sizeof(Sprite));
		sp = SList[nSprites];
		if (sp == NULL) Die("LoadSprites: Malloc error");

		sp->Name = (char *)malloc(strlen(sname)+1);
		if (sp->Name == NULL) Die("LoadSprites: Malloc Error");
		strcpy(sp->Name,sname);
		sp->nFrames = -1;	/* this must come from the file */
		sp->gc = NULL;
		sp->Width = sp->Height = -1;
		nSprites++;
		}
	   else if (!_my_strcasecmp(ch,"endsprite"))
		{
		if (!insprite) Die("LoadSprites: Line %d: Not inside a sprite ",__Current_Line);
		framenum=insprite=0; sp=NULL; SpritesLoaded++;
		}
	   if (!_my_strcasecmp(ch,"framecount"))
		{
		if (!insprite) Die("LoadSprites: Line %d not inside a sprite",__Current_Line);
		sp->nFrames = RequiredNumericToken();
		sp->gc = (GrObject **) malloc(sp->nFrames * sizeof(GrObject *));
		if (sp->gc == NULL) Die("LoadSprite: Malloc error");
		}
	   else if (!_my_strcasecmp(ch,"dimensions"))
		{
		if (!insprite) Die("LoadSprites: Line %d not inside a sprite",__Current_Line);
		sp->Width = RequiredNumericToken();
		sp->Height = RequiredNumericToken();
		}
	   else if (!_my_strcasecmp(ch,"frame"))
		{
		char fname[256];
		int fd,nbytes;

		if (!insprite) Die("LoadSprites: Line %d not inside a sprite",__Current_Line);
		if (framenum >= sp->nFrames) Die("Sprite %s: Too many frames",name);
		sprintf(fname,"%s/%s",SPRITEDIR,RequiredToken());
		sp->gc[framenum] = LoadFrame(fname,sp);
		++framenum;
		}
	   }

	if (insprite) Die("LoadSprites (%s): missing endsprite");
	return SpritesLoaded;
	}

static GrObject *
LoadFrame(char *fname, Sprite *sp)
	{
	FILE *z;
	int i,fd,nbytes;
	char *ch;
	GrObject *g = NULL, *XpmReadFileToGrObject();

	/* locate the suffix */
	for(i=strlen(fname); i && fname[i-1]!='.'; --i);
	if (fname[i-1] != '.') 
	   Die("LoadFram: Frame file %s has no suffix");
	ch = &fname[i];

	if (!_my_strcasecmp(ch,"raw"))
	   {
	   g = GrCreateObject(sp->Width,sp->Height);
	   if (g == NULL) Die("LoadFrame: malloc error");

	   nbytes = sp->Width * sp->Height * 2;
	   z = fopen(fname,"rb");
	   if (fd<0) Die("LoadSprite: Sprite %s: Cannot load frame %s",
			__Current_Sprite,fname);

	   if (fread(g->data,1,nbytes,z) != nbytes)
		Die("LoadSprite: Sprite %s: Short read loading frame %s",
			__Current_Sprite,fname);
	   fclose(z);
	   }
	if (!_my_strcasecmp(ch,"gif"))
	   {
	   g = LoadGif(fname);
	   if (sp->Width<0 && sp->Height<0)
		{sp->Width = g->Width; sp->Height = g->Height;}

	   if (g->Width != sp->Width || g->Height != sp->Height)
		Die("LoadFrame: frame %s has wrong dimensions",fname);
	   }
	if (!_my_strcasecmp(ch,"xpm"))
	   {
	   g = XpmReadFileToGrObject(fname,"rgb.txt");
	   if (g == NULL) Die("XpmReadFileToGrObject: error");

	   if (sp->Width<0 && sp->Height<0)
		{sp->Width = g->Width; sp->Height = g->Height;}

	   if (g->Width != sp->Width || g->Height != sp->Height)
		Die("LoadFrame: frame %s has wrong dimensions",fname);
	   }

	if (g==NULL) Die("LoadFrame: Unrecognised suffix for frame %s",fname);
	return g;
	}

static int
RequiredNumericToken()
	{
	char *p = RequiredToken();
	int i;

	for(i=0; p[i]; ++i)
	   if (!(isdigit(p[i])))
		 Die("Sprite %s, Line %d: \"%s\" is not numeric",
			__Current_Sprite,__Current_Line,p);

	return atoi(p);
	}

static char *
RequiredToken()
	{
	char *p = strtok(NULL,STRTOK_TOKSTR);

	if (p==NULL) Die("Sprite %s: Syntax error on line %d",
		__Current_Sprite, __Current_Line);
	return p;
	}
/*
** Add an Entity at the given absolute coordinates
*/
int
CreateEntity(int SpriteID, int StartFrame, int x, int y, int p)
	{
	int i,w,h,count;
	Entity *e;

	if (!__done_init) InitSprites();
	if (p<0 || p>=NPLANES) Die("CreateEntity: invalid plane %d",p);
	if (SpriteID <0 || SpriteID >= nSprites)
		Die("CreateEntity: invalid SpriteID %d",SpriteID);
	if (Plane[p].count >= SPRITES_PER_PLANE)
		Die("CreateEntity: Too many sprites");
	if (StartFrame<0 || StartFrame>= SList[SpriteID]->nFrames)
		Die("CreateEntity: Sprite %s: Start Frame %d out of range",
			SList[SpriteID]->Name,StartFrame);

	/* Allocate new entity */
	count = Plane[p].count++;
	Plane[p].EList[count] = (Entity *)malloc(sizeof(Entity));
	if (Plane[p].EList[count] == NULL) Die("CreateEntity: Malloc");
	e = Plane[p].EList[count];

	e->data = SList[SpriteID];
	e->cFrame = StartFrame;
	e->x = x;
	e->y = y;
	e->status = RUNNING;
	w = e->data->Width; h = e->data->Height;
	DrawImage(e->data->gc[StartFrame],x,y,0,0,w-1,h-1,GrIMAGE);

	/* Generate an Entity ID */
	return count | (p<<20);
	}

/*
** UpdateEntities() redraws all entities which overlap into the specified
** area. Only the part of the entity which overlaps is redrawn.
** All coords are in Map coordinates (0-64k).
**
*/
void
UpdateEntities(int ax1, int ay1, int ax2, int ay2)
	{
	int i,j;
	int px1,py1,px2,py2,cx1,cy1,cx2,cy2;
	Entity *e;

	if (!__done_init) InitSprites();
	for(i=0; i<NPLANES; ++i)
	   for(j=0; j<Plane[i].count; ++j)
		{
		e = Plane[i].EList[j];
		if (e->status != RUNNING) continue;
	   
	   	px1 = e->x; py1 = e->y;
	   	px2 = px1 + e->data->Width-1;
	   	py2 = py1 + e->data->Height-1;

	   	/* Check for overlap - subregion returned in rectangle c */
	   	if (Overlap_c(ax1,ay1,ax2,ay2,px1,py1,px2,py2,&cx1,&cy1,&cx2,&cy2))
		   DrawImage(e->data->gc[e->cFrame],cx1,cy1,
			cx1 - px1, cy1 - py1,
			(cx1-px1) + (cx2-cx1),
			(cy1-py1) + (cy2-cy1),
			GrIMAGE);
		}
	}

int
RemoveEntity(int eid)
	{
	int i,w,h,pl;
	Entity *e;

	if (!__done_init) InitSprites();
	pl = eid >> 20; eid &= 0xfffff;
	if (pl<0 || pl>=NPLANES) return -1;
	if (eid<0 || eid >= Plane[pl].count) return -2;
	e = Plane[pl].EList[eid];

	w = e->data->Width;
	h = e->data->Height;

	e->status = DELETED;
	return 0;
	}

void
MoveEntity(int eid, int dx, int dy, int flags)
	{
	int i,w,h,x,y;
	int x1,y1,x2,y2;
	int pl;
	Entity *e;

	if (dx==0 && dy==0) return;

	if (!__done_init) InitSprites();
	pl = eid>>20; eid &= 0xfffff;
	e = Plane[pl].EList[eid];

	w = e->data->Width;
	h = e->data->Height;

	x1 = e->x; y1 = e->y; x2 = x1+w-1; y2=y1+h-1;

	if (flags & E_RELATIVE)
	   { x = e->x + dx; y = e->y + dy;}
	else if (flags & E_ABSOLUTE)
	   { x=dx; y=dy;}
	else return;

	if (x < 0 || x+w >= PIX_WIDTH ||
	    y < 0 || y+h >= PIX_HEIGHT) return;

	e->x=x; e->y=y;

	/* Widen our (x1,y1)-(x2,y2) box to include the new location */
	if (dx<0) x1+=dx; else x2+=dx;
	if (dy<0) y1+=dy; else y2+=dy;
	InvalidateRegion(x1,y1,x2,y2);

	/*
	** Increment to the next frame
	*/
	++e->cFrame;
	if (e->cFrame >= e->data->nFrames) e->cFrame=0;
	}

static void
InitSprites()
	{
	int i,j;

	for(i=0; i<NPLANES; ++i)
	   {
	   for(j=0; j<SPRITES_PER_PLANE; ++j)
		Plane[i].EList[j] = NULL;
	   Plane[i].count=0;
	   }

	for(i=0; i<MAX_SPRITES; ++i) SList[i] = NULL;
	nSprites=0; __done_init=1;
	}
