/*
** Map.c
**
** This module manages the game map.
**
** The playing area is 64k pixels on a side. Each cell is 16x16 pixels, so
** there are 2^25 cells on the map.
**
** This is managed by breaking the map into regions of 64x64 cells, giving
** 2^14 cells (16k cells).
**
** Each cell has the following attributes:
**	- a one-byte terrain type
**	- a one-byte image index
**
** API:
**	Create_Terrain(char *f)	:		Creates an empty terrain map f
**	Close_Terrain(char *file):		Saves a terrain map
**	Load_Terrain(char *file):		Loads a terrain map
**	Get_TerrainType(row,col):		Returns the terrain byte
**	Set_TerrainType(row,col,val):		Sets the terrain byte
**	Get_TerrainImageIndex(row,col);		Gets the image index
**	Set_TerrainImageIndex(row,col,val);	Sets the image index
**	Set_TerrainTypeIndex(row,col,type,var);	Sets the image type and index
*/

#include "game.h"

static int MapRegion(int);
static void shuffle_lru(int);
static void flush_dirty_page(int r);

static char File[32] = {0};		/* The current terrain file */
static FILE *Fd = NULL;

#define DEFAULT_CACHE 2
static int REGION_CACHE_SIZE = 	0;

struct reg_p
	{
	short mapped;			/* 1 if this page is mapped */
	int region;			/* The region number for this page */
	unsigned short buffer[REGION_SIZE];
	short dirty[8];			/* 1 if this page slice has been written */
	};

static struct reg_p *rbuf = NULL;
static int *lru_list = NULL;

void
Set_Terrain_Cache(int n)
	{
	int i;

	if (n<2) return;
	if (rbuf != NULL) /* Flush it all out */
	   {
	   for(i=0; i<REGION_CACHE_SIZE; ++i) if (rbuf[i].mapped)
		flush_dirty_page(i);
	   free(rbuf);
	   }

	if (lru_list != NULL) free(lru_list);
	REGION_CACHE_SIZE = n;

	rbuf = (struct reg_p *) malloc(sizeof(struct reg_p) * n);
	lru_list = (int *)malloc(sizeof(int) * n);

	for(i=0; i<REGION_CACHE_SIZE; ++i)
	   {rbuf[i].mapped=0; lru_list[i] = i;}
	}

int
Load_Terrain(char *file)
	{
	int i;

	if (strlen(file) > 30) Die("Create_Terrain: File name too long");
	strcpy(File,file);

	Fd = fopen(File,"rb+");
	if (Fd==NULL) Die("Cannot open terrain file");

	if (!REGION_CACHE_SIZE) Set_Terrain_Cache(DEFAULT_CACHE);

	for(i=0; i<REGION_CACHE_SIZE; ++i)
	   {rbuf[i].mapped=0; lru_list[i] = i;}
	return 0;
	}
	
static void
flush_dirty_page(int r)
	{
	int j;
	long offset;

	for(j=0; j<8; ++j) if (rbuf[r].dirty[j])
	   {
	   offset = rbuf[r].region * REGION_BYTES + j * REGION_SLICE;
	   fseek(Fd,offset,SEEK_SET);
	   if (fwrite(rbuf[r].buffer + ((j*REGION_SLICE)>>1),REGION_SLICE,1,Fd) != 1)
		Die("Error writing region");
	   rbuf[r].dirty[j]=0;
	   }
	return;
	}

int
Save_Terrain()
	{
	long offset;
	int i,j;

	if (!REGION_CACHE_SIZE) Set_Terrain_Cache(DEFAULT_CACHE);

	/* Write out all dirty pages and close the file */
	for(i=0; i<REGION_CACHE_SIZE; ++i) if (rbuf[i].mapped)
	   {
	   flush_dirty_page(i);
	   rbuf[i].mapped = 0;
	   }

	fclose(Fd); Fd=NULL;
	return 0;
	}

int
Create_Terrain(char *file)
	{
	int row,col,i,j;

	if (strlen(file) > 30) Die("Create_Terrain: File name too long");
	strcpy(File,file);

	fclose(fopen(File,"wb"));

	Fd = fopen(File,"rb+");

	if (!REGION_CACHE_SIZE) Set_Terrain_Cache(DEFAULT_CACHE);

	for(i=0; i<REGION_CACHE_SIZE; ++i)
	   {rbuf[i].mapped=0; lru_list[i] = i;
	   for(j=0; j<8; ++j) rbuf[i].dirty[j]=0;
	   }

	return 0;
	}

/*
** The map has dimensions (PIX_WIDTH/CELL_SIZE) * (PIX_HEIGHT/CELL_SIZE)
** cells. Each cell contains a terrain byte specifying the terrain for that
** cell. The terrain is managed by dividing into pages which are 
** (REGION_WIDTH * REGION_HEIGHT) cells.
*/

int
Set_TerrainTypeIndex(int row, int col, unsigned int type, int var)
	{
	int region;
	int rx,ry;
	int rpage;

	if (row<0 || col<0 || row>= CELLS_DOWN || col >= CELLS_ACROSS)
	   return 0;

	if (!REGION_CACHE_SIZE) Set_Terrain_Cache(DEFAULT_CACHE);

	/* Determine the region page for this cell */
	rx = col / REGION_WIDTH;
	ry = row / REGION_HEIGHT;

	region = ry * REGIONS_ACROSS + rx;
	rpage = MapRegion(region);

	/* Now calculate the offset to the word within the page */
	rx = col - rx * REGION_WIDTH;
	ry = row - ry * REGION_HEIGHT;

	ry = ry * REGION_WIDTH + rx;

	/* The terrain type is the low order byte, index is in high byte */
	rbuf[rpage].buffer[ry] = type | (var<<8);
	rbuf[rpage].dirty[(ry<<1) / REGION_SLICE] = 1;
	}

int
Set_TerrainType(int row, int col, unsigned int val)
	{
	int region;
	int rx,ry;
	int rpage;

	if (row<0 || col<0 || row>= CELLS_DOWN || col >= CELLS_ACROSS)
	   return 0;

	if (!REGION_CACHE_SIZE) Set_Terrain_Cache(DEFAULT_CACHE);

	/* Determine the region page for this cell */
	rx = col / REGION_WIDTH;
	ry = row / REGION_HEIGHT;

	region = ry * REGIONS_ACROSS + rx;
	rpage = MapRegion(region);

	/* Now calculate the offset to the word within the page */
	rx = col - rx * REGION_WIDTH;
	ry = row - ry * REGION_HEIGHT;

	ry = ry * REGION_WIDTH + rx;

	/* The terrain type is the low order byte */
	rbuf[rpage].buffer[ry] &= 0xff00;
	rbuf[rpage].buffer[ry] |= (unsigned char)val;
	rbuf[rpage].dirty[(ry<<1) / REGION_SLICE] = 1;
	}

int
Set_TerrainImageIndex(int row, int col, unsigned int val)
	{
	int region;
	int rx,ry;
	int rpage;

	if (row<0 || col<0 || row>= CELLS_DOWN || col >= CELLS_ACROSS)
	   return 0;
	val <<= 8;

	if (!REGION_CACHE_SIZE) Set_Terrain_Cache(DEFAULT_CACHE);

	/* Determine the region page for this cell */
	rx = col / REGION_WIDTH;
	ry = row / REGION_HEIGHT;

	region = ry * REGIONS_ACROSS + rx;
	rpage = MapRegion(region);

	/* Now calculate the offset to the byte within the page */
	rx = col - rx * REGION_WIDTH;
	ry = row - ry * REGION_HEIGHT;

	ry = ry * REGION_WIDTH + rx;

	/* The terrain type is the low order byte */
	rbuf[rpage].buffer[ry] &= 0xff;
	rbuf[rpage].buffer[ry] |= (unsigned int)val;
	rbuf[rpage].dirty[(ry<<1) / REGION_SLICE] = 1;
	}

unsigned int
Get_TerrainType(int row, int col)
	{
	int region;
	int rx,ry;
	int rpage;

	if (row<0 || col<0 || row>= CELLS_DOWN || col >= CELLS_ACROSS)
	   return T_UNDEFINED;

	if (!REGION_CACHE_SIZE) Set_Terrain_Cache(DEFAULT_CACHE);

	/* Determine the region page for this cell */
	rx = (col / REGION_WIDTH);
	ry = (row / REGION_HEIGHT);

	region = ry * REGIONS_ACROSS + rx;
	rpage = MapRegion(region);

	/* Now calculate the offset to the byte within the page */
	rx = col - rx * REGION_WIDTH;
	ry = row - ry * REGION_HEIGHT;

	ry = ry * REGION_WIDTH + rx;
	return (unsigned int) (rbuf[rpage].buffer[ry] & 0xff);
	}

unsigned int
Get_TerrainImageIndex(int row, int col)
	{
	int region;
	int rx,ry;
	int rpage;

	if (row<0 || col<0 || row>= CELLS_DOWN || col >= CELLS_ACROSS)
	   return 0;

	if (!REGION_CACHE_SIZE) Set_Terrain_Cache(DEFAULT_CACHE);

	/* Determine the region page for this cell */
	rx = (col / REGION_WIDTH);
	ry = (row / REGION_HEIGHT);

	region = ry * REGIONS_ACROSS + rx;
	rpage = MapRegion(region);

	/* Now calculate the offset to the byte within the page */
	rx = col - rx * REGION_WIDTH;
	ry = row - ry * REGION_HEIGHT;

	ry = ry * REGION_WIDTH + rx;
	return (unsigned int) ((rbuf[rpage].buffer[ry] & 0xff00) >> 8);
	}
/*
** MapRegion() returns a character pointer to the requested region, which
** is viewed as a rectangle of dimensions REGION_WIDTH * REGION_HEIGHT cells
*/

static int
MapRegion(int r)
	{
	static int last_i = -1;
	static int last_r = -1;
	int i,j,load_region;
	long offset;

	if (Fd == NULL) Die("Attempting to access region in unmapped file");

	/* Optimise access to the same page as last time */
	if (r == last_r) return last_i;

	for(i=0; i<REGION_CACHE_SIZE; ++i)
	   if (rbuf[i].mapped && rbuf[i].region == r)
		{
		last_i = i;
		last_r = r;
		if (lru_list[0] != i) shuffle_lru(i);
		return last_i;
		}

	/* Replace the oldest page */
	i = lru_list[REGION_CACHE_SIZE - 1];

	if (rbuf[i].mapped) flush_dirty_page(i);
	   
	offset = r * REGION_BYTES;
	if (fseek(Fd,offset,SEEK_SET) || fread(rbuf[i].buffer,REGION_BYTES,1,Fd) != 1)
	   {
	   /* Possibly past EOF */
	   for(j=0; j<REGION_SIZE; ++j) rbuf[i].buffer[j]=T_UNDEFINED;
	   }
	rbuf[i].mapped=1;
	for(j=0; j<8; ++j) rbuf[i].dirty[j]=0;

	rbuf[i].region = last_r = r;
	if (lru_list[0] != i) shuffle_lru(i);

	last_i = i; 
	return last_i;
	}

/*
** We have a list of numbers 0 .. N in some order. Locate entry with value
** n and move it to the front of the list
*/
static void
shuffle_lru(int n)
	{
	int i;

	for(i=REGION_CACHE_SIZE-1; i>0; --i) if (lru_list[i] == n) break;
	if (i)
	   {
	   while (i) {lru_list[i] = lru_list[i-1]; --i;}
	   lru_list[0] = n;
	   }
	}
