/* 
   Unix SMB/Netbios implementation.
   Version 1.9.
   read/write to a files_struct
   Copyright (C) Andrew Tridgell 1992-1998
   
   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.
   
   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "includes.h"

extern int DEBUGLEVEL;


/****************************************************************************
seek a file. Try to avoid the seek if possible
****************************************************************************/

SMB_OFF_T seek_file(files_struct *fsp,SMB_OFF_T pos)
{
  SMB_OFF_T offset = 0;
  SMB_OFF_T seek_ret;

  if (fsp->print_file && lp_postscript(fsp->conn->service))
    offset = 3;

  seek_ret = sys_lseek(fsp->fd_ptr->fd,pos+offset,SEEK_SET);

  /*
   * We want to maintain the fiction that we can seek
   * on a fifo for file system purposes. This allows 
   * people to set up UNIX fifo's that feed data to Windows
   * applications. JRA.
   */

  if((seek_ret == -1) && (errno == ESPIPE)) {
    seek_ret = pos+offset;
    errno = 0;
  }

  if((seek_ret == -1) || (seek_ret != pos+offset)) {
    DEBUG(0,("seek_file: sys_lseek failed. Error was %s\n", strerror(errno) ));
    fsp->pos = -1;
    return -1;
  }

  fsp->pos = seek_ret - offset;

  DEBUG(10,("seek_file: requested pos = %.0f, new pos = %.0f\n",
        (double)(pos+offset), (double)fsp->pos ));

  return(fsp->pos);
}

/****************************************************************************
read from a file
****************************************************************************/
ssize_t read_file(files_struct *fsp,char *data,SMB_OFF_T pos,size_t n)
{
  ssize_t ret=0,readret;

#if USE_READ_PREDICTION
  if (!fsp->can_write) {
    ret = read_predict(fsp->fd_ptr->fd,pos,data,NULL,n);

    data += ret;
    n -= ret;
    pos += ret;
  }
#endif

#if WITH_MMAP
  if (fsp->mmap_ptr) {
	  SMB_OFF_T num = (fsp->mmap_size > pos) ? (fsp->mmap_size - pos) : 0;
	  num = MIN(n,num);
	  if (num > 0) {
		  memcpy(data,fsp->mmap_ptr+pos,num);
		  data += num;
		  pos += num;
		  n -= num;
		  ret += num;
	  }
  }
#endif

  flush_write_cache(fsp);

  if (seek_file(fsp,pos) == -1) {
    DEBUG(3,("read_file: Failed to seek to %.0f\n",(double)pos));
    return(ret);
  }
  
  if (n > 0) {
    readret = read(fsp->fd_ptr->fd,data,n);
    if (readret > 0)
      ret += readret;
  }

  return(ret);
}


/* how many write cache buffers have been allocated */
static int allocated_write_caches;

/****************************************************************************
write to a file
****************************************************************************/
ssize_t write_file(files_struct *fsp,char *data,SMB_OFF_T pos, size_t n)
{
  write_cache *wcp = fsp->wcp;

  if (!fsp->can_write) {
    errno = EPERM;
    return(0);
  }

  if (!fsp->modified) {
    SMB_STRUCT_STAT st;
    fsp->modified = True;
    if (sys_fstat(fsp->fd_ptr->fd,&st) == 0) {
      int dosmode = dos_mode(fsp->conn,fsp->fsp_name,&st);
      if (MAP_ARCHIVE(fsp->conn) && !IS_DOS_ARCHIVE(dosmode)) {	
        file_chmod(fsp->conn,fsp->fsp_name,dosmode | aARCH,&st);
      }
    }  
  }

  /* if this is the first write and we are oplocked then setup the write cache */
  if (fsp->granted_oplock && !wcp) {
    setup_write_cache(fsp);
    allocated_write_caches++;
    wcp = fsp->wcp;
  }

  /* if we have active cache and it isn't contiguous then we flush.
   NOTE: There is a small problem with running out of disk .... */
  if (wcp && wcp->data_size && 
      (wcp->offset+wcp->data_size != pos ||
       wcp->data_size+n > wcp->alloc_size)) {
    flush_write_cache(fsp);
  }

  /* handle write cache  - we can cache if the file is oplocked */
  if (wcp) {
    memcpy(wcp->data+wcp->data_size, data, n);
    if (wcp->data_size == 0) wcp->offset = pos;
    wcp->data_size += n;
    inc_num_write_caches();
    return n; /* .... that's a write :) */
  }


  if ((pos != -1) && (seek_file(fsp,pos) == -1))
    return -1;

  return(write_data(fsp->fd_ptr->fd,data,n));
}

/****************************************************************************
 Delete the write cache structure.
****************************************************************************/
void delete_write_cache(files_struct *fsp)
{
  write_cache *wcp = fsp->wcp;

  if(!wcp)
    return;

  allocated_write_caches--;

  SMB_ASSERT(wcp->data_size == 0);

  free(wcp->data);
  free(wcp);

  fsp->wcp = NULL;
}

/****************************************************************************
 Setup the write cache structure.
****************************************************************************/

BOOL setup_write_cache(files_struct *fsp)
{
  write_cache *wcp;

  if (allocated_write_caches >= MAX_WRITE_CACHES) return False;

  if((wcp = (write_cache *)malloc(sizeof(write_cache))) == NULL) {
    DEBUG(0,("setup_write_cache: malloc fail.\n"));
    return False;
  }

  wcp->offset = 0;
  wcp->alloc_size = lp_write_cache_size(SNUM(fsp->conn));
  wcp->data_size = 0;
  if((wcp->data = malloc(wcp->alloc_size)) == NULL) {
    DEBUG(0,("setup_write_cache: malloc fail for buffer size %u.\n",
          (unsigned int)wcp->alloc_size ));
    free(wcp);
    return False;
  }

  fsp->wcp = wcp;

  return True;
}


/*******************************************************************
 Flush a write cache struct to disk.
********************************************************************/

ssize_t flush_write_cache(files_struct *fsp)
{
  write_cache *wcp = fsp->wcp;
  ssize_t ret;

  if(!wcp || !wcp->data_size)
    return 0;

  ret = write_file(fsp, wcp->data, wcp->offset, wcp->data_size);
  wcp->data_size = 0;
  dec_num_write_caches();
  return ret;
}

/*******************************************************************
sync a file
********************************************************************/

void sync_file(connection_struct *conn, files_struct *fsp)
{
#ifdef HAVE_FSYNC
    if(lp_strict_sync(SNUM(conn)) && fsp->fd_ptr != NULL) {
      flush_write_cache(fsp);
      fsync(fsp->fd_ptr->fd);
    }
#endif
}
