# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import re
import pwd
import grp
import sys
import imp
import logging
from os.path import join, exists, abspath, dirname, islink, basename
from gettext import gettext as _

from sugar_server import util, printf
from sugar_server.util import enforce


VERSION = '1.1'

# The specs are a little unclear on the encoding of UUIDs, so be
# flexible in what we accept
MACHINE_SN_RE = re.compile('^([A-Z]{3}[A-F0-9]{8})$')
MACHINE_UUID_RE = re.compile('^([a-fA-F0-9-]{32,40})$')
PUBKEY_RE = re.compile('ssh-(rsa|dss)\s+\S+')

LOGGING_FORMAT = '%(asctime)s %(levelname)s %(name)s: %(message)s'

SERVICES_DIR = join(abspath(dirname(__file__)), 'services')
SERVICES_ALL = sorted([
        _i for _i in os.listdir(SERVICES_DIR) \
                if exists(join(SERVICES_DIR, _i, 'service.py'))])

user = util.Option(
        _('system user and group to start master process'))

keyring_user = util.Option(
        _('system user to start keyring service'))

debug = util.Option(
        _('debug logging level; multiple argument'),
        default=0, type_cast=int, short_option='-D', action='count')

force = util.Option(
        _('force various operations'),
        default=False, type_cast=util.Option.bool_cast, short_option='-f',
        action='store_true')

foreground = util.Option(
        _('Do not send the process into the background'),
        default=False, type_cast=util.Option.bool_cast, short_option='-F',
        action='store_true')

root = util.Option(
        _('path to the root directory with server data'),
        short_option='-r')

services = util.Option(
        _('colon separated list of enabled services, the final list ' \
                'of launched services depends on particular process; ' \
                'empty list means everything'),
        default=set(SERVICES_ALL) - set(['keyring']), short_option='-s',
        type_cast=util.Option.list_cast, type_repr=util.Option.list_repr)

hostname = util.Option(
        _('hostname to listen for incomming connections and spreading it ' \
                'as-is among sugar clients'),
        default='localhost', short_option='-H')

httpd_port = util.Option(
        _('port for incomming HTTP connections for various services'),
        default=8000, type_cast=int)


def init(parser, logname=None):
    """Initilize sugar_server library.

    Needs to be run by all applications that use sugar_server library before
    using any sugar_server functionality.

    :param parser:
        optparse.OptionParser to look for command-line arguments to overwrite
        loaded configuration
    :param logname:
        if specified, file handler will be created to output logs to the file
        with `logname` filename in standard log directory

    """
    if parser is None:
        # Called from tests
        options, args = None, None
    else:
        util.Option.bind(parser, [
                '/etc/sugar-server.conf',
                '~/.config/sugar-server/config',
                # TODO Deperecated, will be removed in 1.1
                '~/.sugar-server.conf',
                ])
        options, args = parser.parse_args()
        util.Option.merge(options)

        if root.value is None:
            root.value = join(os.curdir, 'library')
        root.value = abspath(root.value)

        me = pwd.getpwuid(os.getuid())
        if user.value is None:
            if _run_from_sources():
                user.value = me.pw_name
            else:
                user.value = 'schoolserver'
        if keyring_user.value is None:
            if _run_from_sources():
                keyring_user.value = me.pw_name
            else:
                user.value = 'schoolserver_keyring'

    if not debug.value:
        logging_level = logging.WARNING
    elif debug.value == 1:
        logging_level = logging.INFO
    else:
        logging_level = logging.DEBUG

    if hasattr(options, 'help') and options.help:
        # No need in further routines,
        # application should print the help message and exit
        return options, args

    root_logger = logging.getLogger()
    root_logger.setLevel(logging_level)

    is_setup_command = parser is None or (args and args[0] == 'setup')
    try:
        if is_setup_command:
            enforce(os.geteuid() == 0 or user.value == keyring_user.value,
                    _('Command setup should be launched by root user'))
            _setup(False)
            # Being a root, you should do nothing
            exit(0)
        else:
            _setup(True)
    except Exception:
        message = _('Instance is broken')
        if not is_setup_command:
            message += ', ' + _('run sugar-server with setup command')
        printf.exception(message)
        printf.flush_hints()
        exit(1)

    if logname is not False:
        if logname:
            filename = join(log_path(), '%s.log' % logname)
        else:
            filename = None

        for handler in root_logger.handlers:
            root_logger.removeHandler(handler)

        logging.basicConfig(level=logging_level, filename=filename,
                format=LOGGING_FORMAT)

    return options, args


def service(name, dry_run=False):
    """Service's main Python module.

    :param name:
        service name, i.e., sub-directory name in the `services/` directory
    :returns:
        `service.py` module from service sources

    """
    mod, imported = _import(name, 'service', dry_run)
    if imported:
        logging.debug('Load %s service', name)
    return _Service(mod)


def import_from(mod_name, service_name=None):
    """Import python module from service directory.

    Import will happen starting from the caller's directory. It makes sense
    only for importing service modules from service sources, i.e., when module
    paths are not in sys.path to avoid cross-service imports and, thus, module
    names clashes.

    :param mod_name:
        python module name to import
    :param service_name:
        service name, i.e., sub-directory name in the `services/` directory;
        if omitted, caller's directory will be used
    :returns:
        module object

    """
    if not service_name:
        # pylint: disable-msg=W0212
        frame = sys._getframe(1)
        service_name = basename(dirname(frame.f_globals['__file__']))

    return _import(service_name, mod_name)[0]


def get_logger(name, fmt=None):
    """Get service logger.

    :param name:
        name of logger that will be used for log filename as well
    :param fmt:
        if not `None`, new logger will be have its own format
    :returns:
        logging.Logger object

    """
    logger = logging.getLogger(name)
    if not logger.handlers:
        logfile = join(log_path(), '%s.log' % name)
        handler = logging.FileHandler(logfile)
        if fmt is None:
            fmt = LOGGING_FORMAT
        handler.setFormatter(logging.Formatter(fmt))
        logger.addHandler(handler)
    return logger


def assert_machine_sn(machine_sn):
    enforce(MACHINE_SN_RE.match(machine_sn),
            _('Invalid machine serial number: %s'), machine_sn)


def assert_machine_uuid(uuid):
    enforce(MACHINE_UUID_RE.match(uuid), _('Invalid machine UUID: %s'), uuid)


def assert_pubkey(pubkey):
    enforce(PUBKEY_RE.match(pubkey), _('Invalid publickey: %s'), pubkey)


def home_path(*args):
    return join(root.value, 'home', *args)


def backup_path(*args):
    return join(root.value, 'home', 'backup', *args)


def hashed_backup_path(uid, *args):
    return backup_path(uid[-2:], uid, *args)


def keyring_path(*args):
    return join(root.value, 'home', 'keyring', *args)


def log_path(*args):
    return join(root.value, 'var', 'log', 'sugar-server', *args)


def var_path(*args):
    return join(root.value, 'var', 'sugar-server', *args)


def disk_usage():
    """The storage usage in percents from total numbers.

    :returns:
        the maximum usage between taken bytes and inodes
        on fs where --root is located

    """
    statvfs = _statvfs(root.value)
    if statvfs.f_blocks:
        blocks_used = 100 - statvfs.f_bavail * 100 / statvfs.f_blocks
    else:
        blocks_used = 0
    if statvfs.f_files:
        inodes_used = 100 - statvfs.f_favail * 100 / statvfs.f_files
    else:
        inodes_used = 0
    return max(blocks_used, inodes_used)


def disk_bytes_usage(path=None):
    """How much directory or fs takes in bytes.

    :param path:
        path to the directory to count usage space;
        if `None`, assume entirely fs where --root is located
    :returns:
        size in bytes

    """
    if path is None:
        statvfs = _statvfs(root.value)
        return (statvfs.f_blocks - statvfs.f_bavail) * statvfs.f_bsize
    else:
        return _du(path)


def disk_bytes_total():
    """Whats the storage capacity in bytes.

    :returns:
        size in bytes on the fs where --root is located

    """
    statvfs = _statvfs(root.value)
    return statvfs.f_blocks * statvfs.f_bsize


def disk_bytes_free():
    """How many free space is in bytes.

    :returns:
        free space in bytes on the fs where --root is located

    """
    statvfs = _statvfs(root.value)
    return statvfs.f_bavail * statvfs.f_bsize


def set_statvfs(total, used=0):
    statvfs_file = file(join(root.value, '.statvfs'), 'w')
    print >> statvfs_file, backup_path()
    print >> statvfs_file, total
    print >> statvfs_file, used
    statvfs_file.close()


def unset_statvfs():
    if exists(join(root.value, '.statvfs')):
        os.unlink(join(root.value, '.statvfs'))


def call_crypto(cmd, *args, **kwargs):
    """Run one of bios-crypto utilities.

    :param cmd:
        the name of utility, how it is named in bios-crypto's
        build/ directory in sources tree, to run
    :param args:
        run command arguments
    :param kwargs:
        named arguments to pass to `call` function
    :returns:
        command output, exception otherwise

    """
    cmdline = ['/usr/bin/bc-%s' % cmd] + list([str(i) for i in args])
    return util.assert_call(cmdline, **kwargs)


class _Service(object):

    def __init__(self, mod):
        self.mod = mod

    @property
    def requires_httpd(self):
        for i in dir(self.mod):
            if i.startswith('GET_') or i.startswith('POST_'):
                return True
        else:
            return False

    def httpd_handler(self, name):
        if hasattr(self.mod, name):
            return getattr(self.mod, name)

    def setup(self):
        self._call('setup')

    def start(self):
        self._call('start')

    def stop(self):
        self._call('stop')

    def import_root(self, *args):
        self._call('import_root', *args)

    def import_xs(self, *args):
        self._call('import_xs', *args)

    def _call(self, name, *args):
        if hasattr(self.mod, name):
            getattr(self.mod, name)(*args)


def _import(name, mod, dry_run=False):
    mod_name = '_%s_%s' % (name, mod)
    result = sys.modules.get(mod_name)
    if result is None and not dry_run:
        mod_path = join(SERVICES_DIR, name)
        try:
            fp, pathname, description = imp.find_module(mod, [mod_path])
            try:
                result = imp.load_module(mod_name, fp, pathname, description)
            finally:
                if fp:
                    fp.close()
        except Exception, error:
            util.exception(_('Fail to import module %s from %s service'),
                    mod, name)
            raise RuntimeError(
                    _('Service "%s" is invalid: %s') % (name, error))
        imported = True
    else:
        imported = False
    return result, imported


def _statvfs(path):
    """Support testing workflow on multi processes level."""
    if not exists(join(root.value, '.statvfs')):
        return os.statvfs(path)

    class Statvfs(object):

        f_bsize = 1
        f_blocks = 1
        f_bavail = 1
        f_files = 1
        f_favail = 1

    statvfs = Statvfs()
    statvfs_file = file(join(root.value, '.statvfs'))
    path = statvfs_file.readline().strip()
    statvfs.f_blocks = int(statvfs_file.readline().strip())
    used = int(statvfs_file.readline().strip()) + _du(path, True)
    statvfs.f_bavail = statvfs.f_blocks - used
    return statvfs


def _du(path, force_=False):
    """Support testing workflow on multi processes level."""
    path = abspath(path)

    if not force_ and not exists(join(root.value, '.statvfs')):
        output = util.assert_call(['/usr/bin/du',
                '--bytes', '--summarize', '--one-file-system', path])
        return int(output.split()[0])

    used = 0
    for root_, dirs, files in os.walk(path):
        for i in dirs[:]:
            if islink(join(root_, i)) or i.startswith('.') and root_ == path:
                dirs.remove(i)
        if root_ != path:
            used += len(files)
    return used


def _setup(dry_run):
    for user_name, mode, path in [
            # root's mode should be 0755 since it is a home directory
            # for sugar-server user and SSH will complain for other modes
            (user.value, 0755, root.value),
            (user.value, 0775, join(root.value, 'home')),
            (user.value, 0775, join(root.value, 'share')),
            (user.value, 0775, join(root.value, 'var')),
            (user.value, 0775, var_path()),
            (user.value, 0775, log_path()),
            (keyring_user.value, 0770, keyring_path()),
            ]:
        if not exists(path):
            enforce(not dry_run, _('Directory %s does not exist'), path)
            logging.info(_('Create %s directory'), path)
            os.makedirs(path)

        stat = os.stat(path)
        uid = pwd.getpwnam(user_name).pw_uid
        gid = grp.getgrnam(user.value).gr_gid

        if uid != stat.st_uid or gid != stat.st_gid:
            enforce(not dry_run, _('Directory %s is not owned by %s'),
                    path, user.value)
            logging.info(_('Change owner for %s directory'), path)
            os.chown(path, uid, gid)

        # We need g+w since there are two users, for master and sign process
        if stat.st_mode & 0777 != mode:
            enforce(not dry_run, _('Directory %s access mode is incorrect'),
                    path)
            os.chmod(path, mode)


def _run_from_sources():
    return exists(join(dirname(__file__), '..', '.git'))


def _populate_services_options_and_commands():

    util.Option.seek('main')
    util.Command.seek('main')
    for i in services.value:
        try:
            util.Option.seek(i, _import(i, 'etc')[0])
            util.Command.seek(i, _import(i, 'etc')[0])
        except ImportError:
            pass


_populate_services_options_and_commands()
