# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import cgi
import logging
import urlparse
from SocketServer import TCPServer, ThreadingMixIn
from SimpleHTTPServer import SimpleHTTPRequestHandler
from gettext import gettext as _

from sugar_server import env, misc, util


_MAX_POST_LENGTH = 4096

_access_logger = None
_server = misc.ThreadingTCPServer()


def start():
    global _access_logger
    _access_logger = env.get_logger('httpd', '')
    _access_logger.propagate = False
    _server.start(_Server())


def stop():
    _server.stop()


class _Server(TCPServer, ThreadingMixIn):

    def __init__(self):
        TCPServer.allow_reuse_address = True
        TCPServer.__init__(self,
                (env.hostname.value, env.httpd_port.value),
                _Handler)


class _Handler(SimpleHTTPRequestHandler):

    _send_content = True

    def do_HEAD(self):
        _Handler._send_content = False
        self.do_GET()

    def do_GET(self):
        self._handle_method('GET')

    def do_POST(self):
        if self.headers.get('Content-type') != 'application/json':
            self._reply(None,
                    _('Only application/json content type is allowed'))
            return

        length = self.headers.get('Content-length')
        if not length:
            self._reply(None, _('Content-length is required'))
            return

        length = int(length)
        if length >= _MAX_POST_LENGTH:
            self._reply(None, _('Content-length is too big'))
            return

        data = self.rfile.read(length)
        self._handle_method('POST', util.json.loads(data))

    def log_message(self, fmt, *args):
        request = fmt % args
        message = '%s - - [%s] %s' % \
                (self.address_string(), self.log_date_time_string(), request)
        _access_logger.info(message)

    def _handle_method(self, method, *args):
        url = urlparse.urlparse(self.path)
        # pylint: disable-msg=E1101
        query = dict(cgi.parse_qsl(url.query))
        path = ''.join([i if i.isalnum() else '_' for i in url.path])
        handler = None

        for service in env.services.value:
            service_handler = env.service(service).httpd_handler(method + path)
            if service_handler is None:
                continue
            if handler is None:
                logging.debug(_('Found HTTP handler for %s in %s service'),
                        self.path, service)
                handler = service_handler
            else:
                logging.warning(_('More than one HTTP handlers for %s, ' \
                        'will skip the one from %s service'),
                        self.path, service)

        if handler is None:
            self._reply(None,
                    _('No handlers to serve %s HTTP request') % self.path)
            return

        try:
            reply = handler(query, *args) or {}
            self._reply(reply, None)
        except Exception, error:
            util.exception(_('Cannot process %s request'), self.requestline)
            self._reply(None, error)

    def _reply(self, reply, error):
        if reply is None:
            reply = {'success': 'ERR', 'error': str(error)}
        else:
            reply['success'] = 'OK'
        reply = util.json.dumps(reply)

        self.send_response(200)
        self.send_header('Content-type', 'application/json')
        self.send_header('Content-length', len(reply))
        self.end_headers()

        if reply and _Handler._send_content:
            self.wfile.write(reply)
            self.wfile.flush()
