# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""IPC related routines.

$Repo: git://git.sugarlabs.org/alsroot/codelets.git$
$File: src/ipc.py$
$Data: 2011-09-20$

"""
import os
import time
import fcntl
import errno
import signal
import socket
import logging
import SocketServer
from os.path import exists, dirname
from gettext import gettext as _


_ERROR_MAGIC = 'ERROR:'


class Client(object):
    """Client class for interaction via UNIX socket."""

    def __init__(self, path):
        self._socket = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        try:
            self._socket.connect(path)
        except Exception:
            logging.error(_('Cannot connect to "%s" IPC server'), path)
            raise

    def send_message(self, message):
        self._socket.send(str(message) + '\n')

    def recv_message(self):
        reply = _recv_line(self._socket)
        if reply.startswith(_ERROR_MAGIC):
            raise RuntimeError(reply.split(':', 1)[-1])
        return reply


class Server(SocketServer.UnixStreamServer, SocketServer.ThreadingMixIn):
    """Server class for interaction via UNIX socket."""

    def __init__(self, path, handle_cb):

        class Handler(SocketServer.BaseRequestHandler):

            def handle(self):
                try:
                    handle_cb(self)
                except Exception, error:
                    logging.exception(
                            _('Cannot process request for %s IPC socket'),
                            path)
                    self.send_message(_ERROR_MAGIC + str(error))

            def send_message(self, message):
                self.request.send(str(message) + '\n')

            def recv_message(self):
                return _recv_line(self.request)

        if exists(path):
            os.unlink(path)

        logging.debug('Start socket server on %s', path)
        SocketServer.UnixStreamServer.__init__(self, path, Handler)


def flock(path, timeout=None):
    """Try to exclusively flock the file.

    :param path:
        path to the lock file
    :param timeout:
        even if lock is successfully gained it will be ignored
        if file's mtime is less then `timeout` specified in seconds
    :returns:
        `None` if someone locked it already;
        `False` if `timeout` triggered;
        otherwise locking object, call its `checkpoint()` method
        if the process, that asked for locking, was successfully finished

    """
    path_dir = dirname(path)
    if path_dir and not exists(path_dir):
        logging.debug('Create %s lock directory', path_dir)
        os.makedirs(path_dir)
    lock = file(path, 'a+')

    try:
        fcntl.flock(lock, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except IOError, error:
        lock.close()
        if error.errno in [errno.EACCES, errno.EAGAIN]:
            logging.debug('Lock %s is already acquired', path)
            return None
        raise

    lock = _FLock(lock)

    if lock.stale(timeout):
        logging.debug('Lock %s is acquired successfully', path)
        return lock
    else:
        logging.debug('No need in acquiring %s lock, it is not stale', path)
        return False


def exclusive_access(path):
    """Mutual exclusive lock between processes.

    If lock file is acquired by some process, this process will be killed.

    :param path:
        path to the lock file
    :returns:
        locking object, call its `close()` method to free the lock

    """
    path_dir = dirname(path)
    if path_dir and not exists(path_dir):
        logging.debug('Create %s lock directory', path_dir)
        os.makedirs(path_dir)
    lock = file(path, 'a+')

    existed_pid = lock.readline().strip()
    if existed_pid.isdigit():
        try:
            os.kill(int(existed_pid), signal.SIGTERM)
            logging.debug('Kill %s process that acquired %s lock',
                    existed_pid, path)
        except OSError:
            pass

    fcntl.flock(lock, fcntl.LOCK_EX)
    os.ftruncate(lock.fileno(), 0)
    lock.write(str(os.getpid()))
    lock.flush()

    logging.debug('Lock %s is acquired successfully', path)
    return _FLock(lock, True)


class _FLock(object):

    def __init__(self, lockfile, truncate=False):
        self._truncate = truncate
        self._lockfile = lockfile
        self._lockfile.seek(0, 2)

    def __del__(self):
        self.close()

    def close(self):
        if self._lockfile is None:
            return
        if self._truncate:
            os.ftruncate(self._lockfile.fileno(), 0)
        self._lockfile.close()
        self._lockfile = None

    @property
    def checkpointed(self):
        return self._lockfile.tell()

    def stale(self, timeout):
        shift = int(time.time()) - int(os.stat(self._lockfile.name).st_mtime)
        return not timeout or not self.checkpointed or shift > timeout

    def checkpoint(self):
        if self._lockfile is None:
            return
        self._lockfile.seek(0, 0)
        self._lockfile.write('checkpoint')
        self._lockfile.close()
        self._lockfile = None


def _recv_line(f):
    line = ''
    while True:
        char = f.recv(1)
        if not char or char == '\n':
            break
        line += char
    return line
