# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import datetime
from os.path import join
from gettext import gettext as _

from sugar_server import util
from sugar_server.util import enforce


def hashed_path(machine_sn, *args):
    return join(*(list(args) + [machine_sn[-2:], machine_sn]))


def leases_path(machine_sn):
    return hashed_path(machine_sn, 'leases')


def oats_delegations_path(machine_sn):
    return hashed_path(machine_sn, 'oats-delegations')


def lease_delegations_path(machine_sn):
    return hashed_path(machine_sn, 'lease-delegations')


class Lease(object):

    def __init__(self, content):
        """Parse lease content.

        :param content:
            lease content

        """
        self._content = content
        parts = self.content.split(' ', 4)
        enforce(self.content.endswith('\n') and len(parts) == 5 and \
                parts[0] == 'act01:' and parts[2] == 'K',
                _('Incorrect lease "%s"'), self.content)
        self._machine_sn = parts[1]
        self._expiry = Expiry(parts[3])

    @property
    def content(self):
        return self._content

    @property
    def machine_sn(self):
        return self._machine_sn

    @property
    def expiry(self):
        return self._expiry

    @property
    def expired(self):
        return self._expiry.date < util.utcnow()


class Delegation(object):

    def __init__(self, content):
        """Parse delegation content.

        :param content:
            delegation content

        """
        self._content = content
        parts = self.content.split(' ', 4)
        enforce(self.content.endswith('\n') and len(parts) == 5 and \
                parts[0] == 'del01:' and parts[3] == 'sig02:',
                _('Incorrect delegation "%s"'), self.content)
        self._machine_sn = parts[1]
        self._chain = ' '.join(parts[-2:])

    @property
    def content(self):
        return self._content

    @property
    def machine_sn(self):
        return self._machine_sn

    @property
    def chain(self):
        return self._chain


class Expiry(object):
    """Represent expiration date."""

    FORMAT = '%Y%m%dT%H%M%SZ'

    # pylint: disable-msg=W0212

    def __init__(self, value=None):
        """Initiate `Expiry` value.

        :param value:
            the initial value, might be `None`, `Expiry`, `int` for days and
            `str` for days or full expiry representation

        """
        is_str = isinstance(value, str) or isinstance(value, unicode)
        if value is None:
            self._abs = None
            self._days = None
        elif isinstance(value, Expiry):
            self._abs = value._abs
            self._days = value._days
        elif isinstance(value, datetime.datetime):
            self._abs = value
            self._days = None
        elif is_str and not value.isdigit():
            if value:
                try:
                    self._abs = datetime.datetime.strptime(
                            value, Expiry.FORMAT)
                except ValueError:
                    raise RuntimeError(_('Absolute expiry "%s", should be ' \
                            'in form of "%s"') % (value, Expiry.FORMAT))
            else:
                self._abs = None
            self._days = None
        elif is_str:
            self._abs = None
            self._days = int(value)
        else:
            enforce(isinstance(value, int), _('Expiry days should be integer'))
            self._abs = None
            self._days = value

    @property
    def days(self):
        """How many days remain for expiration date starting from now."""
        if self._days is not None:
            return self._days
        elif self._abs is not None:
            return max(0, (self._abs - util.utcnow()).days)

    @property
    def date(self):
        """The exact expiration date starting from now."""
        if self._abs is not None:
            return self._abs
        elif self._days is not None:
            return util.utcnow() + datetime.timedelta(days=self._days)

    def __str__(self):
        if self.date is None:
            return ''
        else:
            return self.date.strftime(Expiry.FORMAT)

    def __unicode__(self):
        return self.__str__()

    def __repr__(self):
        return self.__str__()

    def __cmp__(self, other):
        return cmp(str(self), str(other))
