# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import logging
import threading
from os.path import join


def ps():
    """The list of running processes.

    :returns:
        list of dicts with process info, at least pid, cmd and args

    """
    result = []
    for pid in [i for i in os.listdir('/proc') if i.isdigit()]:
        cmdline = file(join('/proc', pid, 'cmdline')).read()
        if not cmdline:
            continue
        entry = {'pid': pid,
                 'cmd': cmdline.split()[0],
                 'args': cmdline,
                 }
        result.append(entry)
    return result


class ThreadingTCPServer(object):
    """Convenient wrapper around BaseServer to run it in a thread."""

    def __init__(self):
        self._server = None
        self._thread = None

    def start(self, server):
        """Start processing server in a separated thread.

        :param server:
            BaseServer object to run

        """
        self.stop()
        self._server = server
        logging.debug('Starting up %r TCP server', self._server)
        self._thread = threading.Thread(target=server.serve_forever)
        self._thread.start()

    def stop(self):
        """Stop serving thread.

        Previosly started by `start()` server will be shutting down and serving
        thread joined.

        """
        if self._server is None:
            return
        logging.debug('Shutting down %r TCP server', self._server)
        self._server.shutdown()
        self._thread.join()
        self._server = None
        self._thread = None
