# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys
import signal
import atexit
import logging
from os.path import exists, abspath, dirname
from gettext import gettext as _

from sugar_server import env, httpd
from sugar_server.util import enforce


# XXX hack to use os._exit in unittsets to break it immediately
_process_exit = exit


def start(foreground, logfile=False):
    """Start sugar-server process.

    :param foreground:
        do not daemonize sugar-server process
    :returns:
        exit status; `0` on success and `> 0` otherwise

    """
    pidfile, pid = _check_for_instance()
    if pid:
        logging.warning(_('sugar-server is already run with pid %s'), pid)
        return 1

    log_path = None
    if logfile and not foreground:
        log_path = abspath(env.log_path('%s.log' % _process_name()))
        enforce(os.access(dirname(log_path), os.W_OK))
        _open_log(log_path)

    for i in env.services.value:
        env.service(i).setup()

    if foreground:
        _launch(log_path)
    else:
        _daemonize(pidfile, log_path, _launch)

    return 0


def hup():
    for process in ['sugar-keyring', 'sugar-server']:
        __, pid = _check_for_instance(process)
        if pid:
            os.kill(pid, signal.SIGHUP)
            logging.info(_('Reload %s process'), process)


def stop():
    """Stop sugar-server daemon process.

    :returns:
        exit status; `0` if process was killed and `> 0` otherwise

    """
    __, pid = _check_for_instance()
    if pid:
        os.kill(pid, signal.SIGTERM)
    else:
        logging.warning(_('sugar-server is not run'))
        return 1

    return 0


def started():
    """Is sugar-server process launched in daemon mode."""
    __, pid = _check_for_instance()
    return bool(pid)


def shutdown():
    """Cleanup on exit."""
    for i in env.services.value:
        env.service(i, dry_run=True).stop()
    httpd.stop()


def _launch(log_path):
    logging.info(_('Start sugar-server'))

    stopped = []

    def sigterm_cb(signum, frame):
        if stopped:
            return
        logging.info(_('Got signal %s, will stop sugar-server'), signum)
        stopped.append(True)

    def sighup_cb(signum, frame):
        logging.info(_('Reload on SIGHUP signal'))
        if log_path:
            _open_log(log_path)

    signal.signal(signal.SIGINT, sigterm_cb)
    signal.signal(signal.SIGTERM, sigterm_cb)
    signal.signal(signal.SIGHUP, sighup_cb)

    requires_httpd = False
    for service in env.services.value:
        logging.info(_('Start %s service'), service)
        env.service(service).start()
        requires_httpd = requires_httpd or env.service(service).requires_httpd

    if requires_httpd:
        httpd.start()

    while not stopped:
        signal.pause()


def _process_name():
    if list(env.services.value) == ['keyring']:
        return 'sugar-keyring'
    else:
        return 'sugar-server'


def _check_for_instance(process=None):
    pid = None
    pidfile = env.var_path('%s.pid' % (process or _process_name()))

    if exists(pidfile):
        try:
            pid = int(file(pidfile).read().strip())
            os.getpgid(pid)
        except (ValueError, OSError):
            pid = None

    return pidfile, pid


def _daemonize(pid_path, log_path, doer):
    """Detach a process from the controlling terminal and run in background."""

    pid_path = abspath(pid_path)
    enforce(os.access(dirname(pid_path), os.W_OK))

    if os.fork() > 0:
        # Exit parent of the first child
        return

    # Decouple from parent environment
    os.chdir(os.sep)
    os.setsid()

    if os.fork() > 0:
        # Exit from second parent
        # pylint: disable-msg=W0212
        os._exit(0)

    # Redirect standard file descriptors
    if not sys.stdin.closed:
        stdin = file('/dev/null')
        os.dup2(stdin.fileno(), sys.stdin.fileno())

    pidfile = file(pid_path, 'w')
    pidfile.write(str(os.getpid()))
    pidfile.close()
    atexit.register(lambda: os.remove(pid_path))

    try:
        doer(log_path)
        status = 0
    except Exception:
        logging.exception(_('Abort sugar-server due to error'))
        status = 1

    shutdown()
    _process_exit(status)


def _open_log(log_path):
    sys.stdout.flush()
    sys.stderr.flush()
    logfile = file(log_path, 'a+')
    os.dup2(logfile.fileno(), sys.stdout.fileno())
    os.dup2(logfile.fileno(), sys.stderr.fileno())
    logfile.close()
