# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import json
import time
import fcntl
import logging
from os.path import join, exists, isabs
from gettext import gettext as _

from sugar_server import env, util


_DEFAULT_KEYS = {
        'users': 'uid',
        'machines': 'machine_sn',
        }


def mtime(table):
    """Last modification time of registry table.

    :param table:
        registry table name

    """
    if not exists(_root(table)):
        return 0
    else:
        return os.stat(_root(table)).st_mtime


def find(table, **kwargs):
    """Find items using optional field values.

    By default, if there are no specified arguments, all existing items
    will be returned. Add arguments, with names equal to particular field,
    to minimize the resulting list.

    :param table:
        registry table name
    :kwargs:
        optional list of field values to find for
    :returns:
        list of dictionaries with found items

    """
    for __, __, files in os.walk(_root(table)):
        for an_id in files:
            data = get(table, an_id)
            for field, value in kwargs.items():
                if data.get(field) != value:
                    break
            else:
                yield data


def get(table, key):
    """Get particular item by key value.

    :param table:
        registry table name
    :param key:
        key value to get item for
    :returns:
        dictionary with item values, if it was found;
        emtpty directory otherwise

    """
    if not key:
        return {}
    path = _path(table, key)
    if not exists(path):
        return {}
    try:
        return dict(json.loads(file(path).read()))
    except Exception, error:
        raise RuntimeError(_('Invalid registry item for %s: %s') % \
                (path, error))


def update(table, key, **kwargs):
    """Update item value for specified key.

    :param table:
        registry table name
    :param key:
        key value of item to update
    :returns:
        `True` if new item was inserted

    """
    path = _path(table, key)
    lock = file(path, 'a+')
    fcntl.flock(lock, fcntl.LOCK_EX)
    try:
        item_data = {}
        file_data = lock.read()
        if file_data:
            try:
                item_data = dict(json.loads(file_data))
            except Exception:
                logging.warning(_('Invalid "%s" registry item will be ' \
                        'overwritten'), key)

        is_new = not item_data
        item_data.update(kwargs)
        if table in _DEFAULT_KEYS:
            item_data[_DEFAULT_KEYS[table]] = key

        item_file = util.new_file(path)
        item_file.write(json.dumps(item_data))
        item_file.close()

        _set_mtime(table)
        return is_new
    finally:
        lock.close()


def _root(table, *args):
    if isabs(table):
        return join(table, *args)
    else:
        root_path = env.home_path('registry')
        return join(root_path, table, *args)


def _path(table, key):
    # Not using first numbers since SNs have them the same most of the time
    result_dir = _root(table, key[-2:])
    if not exists(result_dir):
        os.makedirs(result_dir)
    return join(result_dir, key)


def _set_mtime(table):
    timestamp = time.time()
    os.utime(_root(table), (timestamp, timestamp))
