# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import hashlib
from os.path import exists, dirname, join
from gettext import gettext as _

from sugar_server import env, ipc, leases
from sugar_server.util import enforce


etc = env.import_from('etc')

_logger = env.get_logger('activation')


class Lease(leases.Lease):

    def __init__(self, content, machine_sn=None, delegated=False):
        """Parse lease content.

        :param content:
            lease content
        :param machine_sn:
            if specified, compare it with serial number parsed from
            lease content to make sure if lease is valid

        """
        leases.Lease.__init__(self, content)
        enforce(not machine_sn or self.machine_sn == machine_sn,
                _('Found lease is not for %s'), machine_sn)
        self._registry = None
        self._delegated = delegated

    @property
    def delegated(self):
        return self._delegated


class Delegation(leases.Delegation):

    def __init__(self, content, machine_sn=None):
        """Parse delegation content.

        :param content:
            delegation content
        :param machine_sn:
            if specified, compare it with serial number parsed from
            lease content to make sure if lease is valid

        """
        leases.Delegation.__init__(self, content)
        enforce(not machine_sn or self.machine_sn == machine_sn,
                _('Found delegation is not for %s'), machine_sn)


def get(machine_sn, expiry=None):
    if machine_sn == etc.TEST_SN:
        _logger.debug('Got %s test lease request', machine_sn)
        lease_path = join(dirname(__file__), 'tests', machine_sn)
    else:
        lease_path = etc.leases_path(machine_sn)
    _logger.debug('Lease path is %s', lease_path)

    if exists(lease_path):
        try:
            lease = Lease(file(lease_path).read(), machine_sn)
            if lease.expired:
                _logger.info(_('Pre-existing lease expiried for %s, ' \
                        'will delegate'), machine_sn)
            else:
                _logger.info(_('Found pre-existing lease for %s'), machine_sn)
                return lease
        except Exception:
            _logger.exception(_('Incorrect pre-existing lease, ' \
                    'will try to gelegate new one'))
    else:
        enforce(etc.delegation.value,
                _('No pre-existing lease and delegation is disabled'))
        _logger.info(_('No pre-existing lease for %s, will delegate'),
                machine_sn)

    if expiry is None:
        _logger.info(_('Use default lease expiry for %s, %s'),
                machine_sn, etc.default_expiry.value)
        expiry = etc.default_expiry.value

    lease = delegate(machine_sn, expiry)
    return Lease(lease, machine_sn, delegated=True)


def get_stolen_status(uuid, stolen, nonce):
    if stolen:
        payload = '%s:%s:STOLEN' % (uuid, nonce)
    else:
        payload = '%s:%s' % (uuid, nonce)
    return hashlib.sha256(payload).hexdigest()


def sign_time(machine_sn, nonce, expiry=1):
    # XXX hardcoded in xs-activation/oat.py sources
    now = '00110529T135103Z'
    payload = '%s:%s' % (now, nonce)
    signed = sign(machine_sn, expiry, payload)
    return 'time01: %s %s' % (now, signed)


def sign(machine_sn, expiry, data):
    oats_path = etc.oats_delegations_path(machine_sn)
    enforce(oats_path, _('No OAT delegation for %s'), machine_sn)
    oats_delegation = Delegation(file(oats_path).read(), machine_sn)

    args = {'machine_sn': machine_sn,
            'expiry': leases.Expiry(expiry),
            'data': data,
            }
    payload = '%(machine_sn)s:%(expiry)s:%(data)s' % args
    signed = _call_keyring('sign', expiry, payload)
    _sig02_prefix_to_skip_, signed = signed.split(' ', 1)

    return '%s %s' % (oats_delegation.chain, signed)


def delegate(machine_sn, expiry):
    expiry = leases.Expiry(expiry)

    ld_path = etc.lease_delegations_path(machine_sn)
    enforce(exists(ld_path), _('No lease delegation for %s'), machine_sn)

    ld_fields = file(ld_path).read().split(' ', 3)
    enforce(len(ld_fields) == 4 and ld_fields[0] == 'del01:',
            _('Incorrect delegation header in %s for %s'), ld_path, machine_sn)

    ld_sn, machine_uuid, chain = ld_fields[1:]
    enforce(ld_sn == machine_sn,
            _('Wrong machine serial number in %s, it should be %s'),
            ld_path, machine_sn)

    if etc.max_expiry.value is not None and expiry > etc.max_expiry.value:
        _logger.info(_('Specified expiry for %s, %s, is too long, ' \
                'will use the possible maximum, %s'),
                machine_sn, expiry, etc.max_expiry.value)
        expiry = etc.max_expiry.value

    args = {'sn': machine_sn,
            'uuid': machine_uuid,
            'expiry': str(expiry),
            'chain': chain.split('\n')[0],
            }
    prefix = 'act01: %(sn)s K %(expiry)s %(chain)s' % args
    payload = '%(sn)s:%(expiry)s:%(sn)s:%(uuid)s:K:%(expiry)s' % args

    return prefix + _call_keyring('sign', expiry, payload)[6:] + '\n'


def _call_keyring(*args):
    client = ipc.Client(env.var_path('keyring'))
    for i in args:
        client.send_message(i)
    return client.recv_message()
