# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import re
import time
import SocketServer
from gettext import gettext as _

from sugar_server import env, misc, registry
from sugar_server.util import enforce


etc = env.import_from('etc')
crypto = env.import_from('crypto')

_MAX_REQUEST_SIZE = 512
_COMMAND_RE = re.compile('^([a-zA-Z0-9]+): (.*)$')

_logger = env.get_logger('activation')
_service = misc.ThreadingTCPServer()


def start():
    _service.start(_Server())


def stop():
    _service.stop()


class Handler(SocketServer.BaseRequestHandler):

    def handle(self):
        # The request cannot be bigger
        request = self.request.recv(_MAX_REQUEST_SIZE)
        try:
            self.do_handle(request)
        except Exception:
            _logger.exception(_('Cannot process request "%s"'), request)
            self.request.send('UNKNOWN\n')

    def do_handle(self, request):
        match = env.MACHINE_SN_RE.match(request)
        if match is not None:
            reply = _process_lease_request(match.group(1))
        else:
            match = _COMMAND_RE.match(request)
            if match is not None:
                command, args = match.groups()
                if command == 'time01':
                    reply = _process_time_request(args)
                else:
                    raise RuntimeError(_('Unknown command %s') % command)
            else:
                raise RuntimeError(_('Cannot parse request "%s"') % request)
        self.request.send(reply)


def GET_client_lease(query):
    enforce('machine_sn' in query,
            _('Query parameter "machine_sn" needs to be passed with request'))
    lease = _process_lease_request(query['machine_sn'])
    enforce(lease != 'STOLEN',
            _('Requested lease is stated as stolen'))
    return {'lease': lease}


class _Server(SocketServer.TCPServer, SocketServer.ThreadingMixIn):

    def __init__(self):
        SocketServer.TCPServer.allow_reuse_address = True
        SocketServer.TCPServer.__init__(self,
                (env.hostname.value, etc.activation_port.value),
                Handler)


def _process_lease_request(machine_sn):
    timestamp = int(time.time())
    machine = registry.get('machines', machine_sn)
    machine['activation_requested'] = timestamp

    if machine.get('stolen'):
        _logger.info(_('Requested lease for %s is stolen'), machine_sn)
        reply = 'STOLEN'
        machine['revealed'] = timestamp
    else:
        _logger.info(_('Process lease request for %s'), machine_sn)
        lease = crypto.get(machine_sn)
        reply = lease.content
        machine['activated'] = timestamp

    registry.update('machines', machine_sn, **machine)
    return reply


def _process_time_request(args_str):
    args = args_str.split()
    enforce(len(args) == 2, _('Wrong number of arguments for time01 command'))

    machine_sn, nonce = args
    env.assert_machine_sn(machine_sn)
    enforce(len(nonce) == 22,
            _('time01 command argument "nonce", should be 22 chars long'))

    return crypto.sign_time(machine_sn, nonce)
