# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys
import shutil
import logging
from datetime import datetime
from optparse import OptionParser
from os.path import exists, join, lexists, basename, abspath, dirname

from sugar_server import util, env
from sugar_server.util import enforce


etc = env.import_from('etc')

RSYNC = '/usr/bin/rsync'

# ds-backup clients touch completion flag file, that is unneeded here
# but it should be handled to not break client logic
DST_DEV_NULL = '.dev.null'


def cook_rsync_command(root):
    enforce('SSH_ORIGINAL_COMMAND' in os.environ,
            'Envrionment variable SSH_ORIGINAL_COMMAND is not exported, ' \
            'most likelly you run program not from .ssh/authorized_keys')
    ssh_original_cmd = os.environ['SSH_ORIGINAL_COMMAND'].split()
    logging.debug('Got rsync request: %s', ssh_original_cmd)
    enforce(ssh_original_cmd and basename(ssh_original_cmd[0]) == 'rsync',
            'Only rsync command is supported')

    root = abspath(join(root, 'current'))
    if not exists(root):
        logging.info('Create backup root %s', root)
        os.makedirs(root)

    sys.argv[:] = [RSYNC] + ssh_original_cmd[1:]

    parser = OptionParser()
    parser.add_option('', '--server',
            help='See rsync(8)', action='store_true')
    parser.add_option('', '--sender',
            help='See rsync(8)', action='store_true')
    parser.add_option('-v', '--verbose',
            help='See rsync(8)', action='store_true')
    parser.add_option('-l', '--links',
            help='See rsync(8)', action='store_true')
    parser.add_option('-t', '--times',
            help='See rsync(8)', action='store_true')
    parser.add_option('-r', '--recursive',
            help='See rsync(8)', action='store_true')
    parser.add_option('-z', '--compress',
            help='See rsync(8)', action='store_true')
    parser.add_option('-i', '--itemize-changes',
            help='See rsync(8)', action='store_true')
    parser.add_option('-L', '--copy-links',
            help='See rsync(8)', action='store_true')
    parser.add_option('-s', '--protect-args',
            help='See rsync(8)', action='store_true')
    parser.add_option('-f', '',
            help='See rsync(8)', action='store_true')
    parser.add_option('', '--timeout',
            help='See rsync(8)')
    parser.add_option('', '--delete',
            help='See rsync(8)', action='store_true')
    parser.add_option('', '--partial',
            help='See rsync(8)', action='store_true')
    parser.add_option('-T', '--temp-dir',
            help='See rsync(8)')
    parser.add_option('-e', '--rsh',
            help='See rsync(8)')
    parser.add_option('-n', '--dry-run',
            help='See rsync(8)', action='store_true')
    parser.add_option('', '--log-format',
            help='See rsync(8)')
    parser.add_option('', '--partial-dir',
            help='See rsync(8)')

    options, args = parser.parse_args()

    enforce(options.server, 'Argument --server should be passed')
    enforce(len(args) == 2, 'Pass src and dst directories only')
    src, dst = args
    enforce(src == '.', 'src should be "."')

    dst_parts = dst.lstrip(os.sep).split(os.sep)
    if dst_parts[0] == 'current':
        dst_parts.pop(0)
    elif dst_parts[0] == 'datastore-current':
        # Support XS's paths
        dst_parts.pop(0)
    else:
        try:
            date = datetime.strptime(dst_parts[0], etc.BACKUP_DIRNAME)
            root = join(dirname(root), date.strftime(etc.BACKUP_DIRNAME))
            dst_parts.pop(0)
        except ValueError:
            logging.info('Requested remote path does not start with ' \
                    '"current", fixed')
    dst = abspath(join(root, *dst_parts))
    enforce(dst.startswith(root),
            'Trying to get access out of personal directory')
    # Revert the last slash or make sure that while accessing to the top
    # directory, destination will contain the last slash
    if dst_parts and not dst_parts[-1] or dst == root:
        dst += os.sep

    argv = sys.argv[:1] + ['--server']
    if options.sender:
        argv.append('--sender')
    if options.verbose:
        argv.append('--verbose')
    if options.rsh:
        argv.append('--rsh')
        argv.append(options.rsh)
    if options.links:
        argv.append('--links')
    if options.times:
        argv.append('--times')
    if options.recursive:
        argv.append('--recursive')
    if options.compress:
        argv.append('--compress')
    if options.itemize_changes:
        argv.append('--itemize-changes')
    if options.copy_links:
        argv.append('--copy-links')
    if options.protect_args:
        argv.append('--protect-args')
    if options.timeout:
        argv.append('--timeout=%s' % options.timeout)
    if options.delete:
        argv.append('--delete')
    if options.partial:
        argv.append('--partial')
    if options.f:
        argv.append('-f')
    if options.dry_run:
        argv.append('--dry-run')
    if options.log_format:
        argv.append('--log-format="%s"' % options.log_format)
    if options.partial_dir:
        argv.append('--partial-dir="%s"' % options.partial_dir)

    argv.append(src)
    argv.append(dst)

    logging.debug('Converted rsync command: %r', argv)

    return argv, options.dry_run


def postprocess(root):
    dev_null = join(root, 'current', DST_DEV_NULL)
    if exists(dev_null):
        os.unlink(dev_null)
        return

    backup_dirname = util.utcnow().strftime(etc.BACKUP_DIRNAME)
    dst = join(root, backup_dirname)
    if exists(dst):
        shutil.rmtree(dst, ignore_errors=True)

    try:
        util.cptree(join(root, 'current'), dst)

        latest_path = join(root, 'latest')
        if lexists(latest_path):
            os.unlink(latest_path)
        os.symlink(backup_dirname, latest_path)

        trimmer = env.import_from('trimmer', 'backup')
        trimmer.trim()
    except Exception:
        logging.error('Error while postprocessing rsynced backup, ' \
                'remove remains')
        shutil.rmtree(dst, ignore_errors=True)
        if not exists(latest_path):
            os.unlink(latest_path)
            os.symlink('current', latest_path)
        raise
