# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import pwd
import shutil
import logging
import threading
from ConfigParser import ConfigParser
from datetime import datetime
from os.path import join, exists, basename, abspath, dirname, isdir
from gettext import gettext as _

from sugar_server import env, misc, printf, util, registry
from sugar_server.util import enforce


etc = env.import_from('etc')
trimmer = env.import_from('trimmer')
rsync = env.import_from('rsync')


_LOADAVG_MAX = 5
_CONCURRENT_RSYNCS_MAX = 10
_RSYNC = '/usr/bin/rsync'
_XS_BACKUP_DIRNAME = 'datastore-%Y-%m-%d_%H:%M'

_logger = env.get_logger('backup')

_service_cond = threading.Condition()
_service_exit = False


def setup():
    if pwd.getpwnam(env.user.value).pw_dir != env.root.value:
        logging.warning(_('Home directory for user "%s" should be "%s" ' \
                'to let users do backups via SSH'), env.user, env.root)

    if not exists(env.backup_path()):
        _logger.info(_('Create %s backup directory'), env.backup_path())
        os.makedirs(env.backup_path())
    if os.stat(env.backup_path()).st_mode & 0777 != 0700:
        _logger.info(_('Change access mode for %s backup directory'),
                env.backup_path())
        os.chmod(env.backup_path(), 0700)

    rsync_src = join(env.SERVICES_DIR, 'backup', 'sugar-server-rsync')
    rsync_dst = env.backup_path('sugar-server-rsync')
    rsync_src_stat = os.stat(rsync_src)

    if not exists(rsync_dst) or \
            rsync_src_stat.st_mtime != os.stat(rsync_dst).st_mtime:
        _logger.info(_('Update sugar-server-rsync rsync wrapper script'))

        rsync_txt = []
        for line in file(rsync_src).readlines():
            pypath = abspath(join(env.SERVICES_DIR, '..', '..'))
            rsync_txt.append(line.replace('@SUGAR_SERVER_PYTHONPATH@', pypath))

        rsync_file = file(rsync_dst, 'w')
        rsync_file.write(''.join(rsync_txt))
        rsync_file.close()
        os.chmod(rsync_dst, 0755)
        os.utime(rsync_dst, (rsync_src_stat.st_mtime, rsync_src_stat.st_mtime))

    rsync_conf = ConfigParser()
    for prop in util.Option.items.values():
        if not rsync_conf.has_section(prop.section):
            rsync_conf.add_section(prop.section)
        rsync_conf.set(prop.section, prop.name, str(prop))
    rsync_conf_file = file(env.backup_path('sugar-server-rsync.conf'), 'w')
    rsync_conf.write(rsync_conf_file)
    rsync_conf_file.close()


def start():

    def service():
        _service_cond.acquire()
        while True:
            try:
                trimmer.trim()
            except Exception:
                _logger.exception(_('Cannot process trimming'))
            _service_cond.wait(etc.trim_timeout.value)
            if _service_exit:
                break
        _service_cond.release()

    threading.Thread(target=service).start()


def stop():
    _wakeup(True)


def import_root(root):
    root = join(root, 'home', 'backup')
    enforce(root != env.backup_path())

    for src_path in trimmer.walk_backups(root):
        if exists(join(src_path, 'current')):
            _import_dir(src_path, basename(src_path))


def import_xs(root):

    def fix_dst(dst_path):
        max_date = None

        for filename in os.listdir(dst_path):
            path = join(dst_path, filename)
            try:
                date = datetime.strptime(filename, _XS_BACKUP_DIRNAME)
            except ValueError:
                date = None

            if date is not None:
                if max_date is None:
                    max_date = date
                else:
                    max_date = max(max_date, date)

                new_filename = date.strftime(etc.BACKUP_DIRNAME)
                os.rename(path, join(dst_path, new_filename))
            elif filename == 'datastore-current':
                os.rename(path, join(dst_path, 'current'))
            elif isdir(path):
                shutil.rmtree(path, ignore_errors=True)
            else:
                os.unlink(path)

        if max_date is not None:
            filename = max_date.strftime(etc.BACKUP_DIRNAME)
            os.symlink(filename, join(dst_path, 'latest'))

    root = join(root, 'users')

    for sn in os.listdir(root):
        src_path = join(root, sn)
        if not exists(join(src_path, 'datastore-current')) or \
                env.MACHINE_SN_RE.match(sn) is None:
            continue

        machine = registry.get('machines', sn)
        if not machine.get('uid'):
            _logger.debug('Skip importing %s, not registered', sn)
            continue

        _import_dir(src_path, machine['uid'], fix_dst)


def GET_client_backup(query):
    enforce('uid' in query,
            _('Query parameter "uid" needs to be passed with request'))
    enforce(registry.get('users', query['uid']),
            _('The %(uid)s is not registered') % query)
    return {'accepted': flow_opened()}


def flow_opened():
    loadavg = os.getloadavg()[0]
    if loadavg > _LOADAVG_MAX:
        _logger.info(_('The system load avarage is too high, %s > %s. ' \
                'Will deny it'), loadavg, _LOADAVG_MAX)
        return False

    if env.disk_usage() >= etc.hard_quota.value:
        _logger.warning(_('No free storage space to accept new backup, ' \
                '%s >= %s'), env.disk_usage(), etc.hard_quota.value)
        _wakeup()
        return False

    concurrent_rsyncs = \
            len([i for i in misc.ps() if basename(i['cmd']) == 'rsync'])
    if concurrent_rsyncs > _CONCURRENT_RSYNCS_MAX:
        _logger.info(_('Too many rsync concurrent sessions, %s > %s. ' \
                'Will deny backup'), concurrent_rsyncs, _CONCURRENT_RSYNCS_MAX)
        return False

    return True


def CMD_trim(args):
    result = trimmer.trim(env.force.value)

    if result is trimmer.NO_NEED:
        printf.info(_('No need in trimming at all'))
    elif result is trimmer.ONGOING:
        printf.info(_('Trimming is already started by someone else'))
    elif result is trimmer.SKIPPED:
        printf.info(_('Regular trimming already finished, no need ' \
                'this time. Pass --force argument to force trimming'))
    elif result is trimmer.SUCCESS:
        printf.info(_('Trimming finished successfully'))


def _wakeup(is_exit=None):
    global _service_exit
    _service_cond.acquire()
    _service_exit = is_exit
    _service_cond.notify()
    _service_cond.release()


def _import_dir(src_path, uid, fix_cb=None):
    dst_path = env.hashed_backup_path(uid)
    if exists(dst_path):
        _logger.debug('Skip importing %s, backup exists', uid)
        return

    tmp_dst_path = dst_path + '.import'
    try:
        _logger.info(_('Import backup directory for %s'), uid)
        _logger.debug('Copy backup directory %s to %s', src_path, tmp_dst_path)

        if not exists(dirname(tmp_dst_path)):
            os.makedirs(dirname(tmp_dst_path))
        util.assert_call(
                [_RSYNC, '-rltH', src_path + os.sep, tmp_dst_path])

        if fix_cb is not None:
            fix_cb(tmp_dst_path)

        os.rename(tmp_dst_path, dst_path)

    except Exception:
        util.exception(_logger,
                _('Cannot import backup directory for %s'), uid)
        shutil.rmtree(tmp_dst_path, ignore_errors=True)
