# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import bisect
import shutil
import datetime
from os.path import join, isdir, exists, basename
from gettext import gettext as _

from sugar_server import env, ipc


etc = env.import_from('etc')

NO_NEED = 0
ONGOING = 1
SKIPPED = 2
SUCCESS = 3

logger = env.get_logger('backup')


def trim(force=False):
    """Trim backups to conform soft_quota for disk space.

    :returns:
        `NO_NEED`, no need in trimming at all;
        `ONGOING`, trimming is already started by someone else;
        `SKIPPED`, regular trimming already finished, no need this time;
        `SUCCESS`, trimming finished successfully

    """
    if env.disk_usage() <= etc.soft_quota.value:
        return NO_NEED

    if force or env.disk_usage() >= etc.hard_quota.value:
        # Force to lock despite of doing trimming already
        timeout = None
    else:
        timeout = etc.trim_timeout.value
    lock = ipc.flock(env.var_path('trim-backup.lock'), timeout)
    if lock is None:
        return ONGOING
    elif lock is False:
        return SKIPPED

    users_dirs = []
    users_bytes_usage = 0

    for path in walk_backups(env.backup_path()):
        bytes_usage = env.disk_bytes_usage(path)
        users_bytes_usage += bytes_usage
        users_dirs.append((path, bytes_usage))

    if not users_dirs:
        return NO_NEED

    bytes_to_free = env.disk_bytes_usage() - \
            (etc.soft_quota.value * env.disk_bytes_total()) / 100
    per_user_bytes_usage = \
            max(0, (users_bytes_usage - bytes_to_free) / len(users_dirs))

    logger.info(_('Start trimming backups to free %s bytes keeping %s ' \
            'bytes per user'), bytes_to_free, per_user_bytes_usage)

    bytes_free = env.disk_bytes_free()
    for root, bytes_usage in users_dirs:
        if bytes_usage <= per_user_bytes_usage:
            continue

        latest_link_path = join(root, 'latest')
        if exists(latest_link_path):
            latest_basename = os.readlink(latest_link_path)
        else:
            latest_basename = None

        for backup_path in walk(root):
            logger.info(_('Wiping out %s backup'), backup_path)

            if latest_basename and basename(backup_path) == latest_basename:
                os.unlink(latest_link_path)
                os.symlink('current', latest_link_path)

            shutil.rmtree(backup_path)

            # Decrement counted user directory size by the freed size
            new_bytes_free = env.disk_bytes_free()
            bytes_usage -= (new_bytes_free - bytes_free)
            bytes_free = new_bytes_free

            if bytes_usage <= per_user_bytes_usage:
                break

    lock.checkpoint()
    logger.info(_('Trimming finished successfully'))

    return SUCCESS


def walk_backups(top):
    # pylint: disable-msg=E1101
    __, dirs, __ = os.walk(top).next()
    for root in dirs:
        root = join(top, root)
        if not isdir(root):
            continue
        for i in os.listdir(root):
            yield join(root, i)


def walk(root):
    """Iterate top-level sub-directories in user's backup root.

    The return order is from the most useless to the most usefull, i.e.,
    ready for trimming process.

    """
    backups = []

    for i in os.listdir(root):
        path = join(root, i)
        if not isdir(path):
            continue
        try:
            date = datetime.datetime.strptime(i, etc.BACKUP_DIRNAME)
        except ValueError:
            continue
        bisect.insort_left(backups, (date, path))

    if not backups:
        return
    latest_date, __ = backups[-1]

    # At first, remove all daily backups out of trim-daily-limit
    # keeping one backup per month
    monthly_month, monthly_path, monthly_i = None, None, None
    for i, (date, path) in enumerate(backups):
        month = (date.year, date.month)
        if month == monthly_month:
            yield monthly_path
            backups[monthly_i] = (None, None)
        if (latest_date - date).days < etc.trim_daily_limit.value:
            break
        monthly_month = month
        monthly_path = path
        monthly_i = i

    # Then, trim all backups one-by-one from the earliest ones
    for date, path in backups:
        if date:
            yield path
