# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import shutil
import logging
from os.path import exists, dirname
from gettext import gettext as _

from sugar_server import env, ipc, misc, leases
from sugar_server.util import enforce


_private_key_path = env.keyring_path('private', 'server')
_service = misc.ThreadingTCPServer()


def setup():
    # Only signer visible server keys
    key_dir = dirname(_private_key_path)
    if exists(key_dir):
        stat = os.stat(key_dir)
        enforce(stat.st_uid == os.getuid(),
                _('Wrong user for %s directory'), key_dir)
        enforce(stat.st_gid == os.getgid(),
                _('Wrong group for %s directory'), key_dir)
    else:
        logging.info(_('Create private keys directory in %s'), key_dir)
        os.makedirs(key_dir)
    if os.stat(key_dir).st_mode & 0777 != 0700:
        logging.info(_('Change access mode for %s keys directory'), key_dir)
        os.chmod(key_dir, 0700)

    # Check for server keys
    pub_path = _private_key_path + '.public'
    priv_path = _private_key_path + '.private'
    if not exists(pub_path) or not exists(priv_path):
        logging.info(_('Server keys are missed, will generate them'))
        env.call_crypto('makekey', _private_key_path)
        shutil.copy(pub_path, env.keyring_path())
    else:
        stat = os.stat(priv_path)
        enforce(stat.st_uid == os.getuid(), _('Wrong user for %s'), priv_path)
        enforce(stat.st_gid == os.getgid(), _('Wrong group for %s'), priv_path)
    if os.stat(priv_path).st_mode & 0777 != 0600:
        logging.info(_('Change access mode for %s keys file'), priv_path)
        os.chmod(priv_path, 0600)


def start():
    _service.start(ipc.Server(env.var_path('keyring'), handle))


def stop():
    _service.stop()


def handle(handler):

    def recv_expire():
        expiry = leases.Expiry(handler.recv_message())
        enforce(expiry, _('Worng expire argument for act01 command'))
        return expiry

    command = handler.recv_message()

    if command == 'sign':
        expiry = recv_expire()
        payload = handler.recv_message()
        signed = sign(expiry, payload)
        handler.send_message(signed)
    else:
        raise Exception(_('Unknow keyring command, "%s"'), command)


def sign(expiry, data):
    signed = env.call_crypto('sig01', '--v2', leases.Expiry(expiry),
            '--fullkey', 'sha256', _private_key_path, stdin=data)
    return signed
