# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import pwd
import grp
import time
from os.path import join, exists, dirname
from gettext import gettext as _

from sugar_server import env, util, registry
from sugar_server.util import enforce


_logger = env.get_logger('registry')
_directory = None
_authorized_keys_path = join(env.root.value, '.ssh', 'authorized_keys')


def setup():
    backup_ssh_path = dirname(_authorized_keys_path)
    if not exists(backup_ssh_path):
        _logger.info(_('Create %s SSH directory for backup user'),
                backup_ssh_path)
        os.makedirs(backup_ssh_path)

    stat = os.stat(backup_ssh_path)
    if stat.st_mode & 0777 != 0700:
        _logger.info(_('Change access mode of %s SSH directory ' \
                'for backup user'), backup_ssh_path)
        os.chmod(backup_ssh_path, 0700)

    uid = pwd.getpwnam(env.user.value).pw_uid
    gid = grp.getgrnam(env.user.value).gr_gid
    enforce(stat.st_uid == uid and stat.st_gid == gid,
            _('Wrong user for %s, it should be %s'), backup_ssh_path, env.user)

    if not exists(_authorized_keys_path):
        create_keys()
    else:
        stat = os.stat(_authorized_keys_path)
        if stat.st_mode & 0777 != 0600 or \
                stat.st_uid != uid or stat.st_gid != gid or \
                stat.st_mtime != _registry_mtime():
            create_keys()


def close():
    if exists(_authorized_keys_path):
        mtime = _registry_mtime()
        os.utime(_authorized_keys_path, (mtime, mtime))


def register(uid, nickname, pubkey, machine_sn, machine_uuid):
    if machine_sn:
        env.assert_machine_sn(machine_sn)
    if machine_uuid:
        env.assert_machine_uuid(machine_uuid)
    enforce('\n' not in nickname, _('Invalid nickname: %s'), nickname)
    # TODO should precise match pubkeys
    enforce(pubkey.strip() and '\n' not in pubkey,
            _('Invalid public key: %s'), pubkey)

    if machine_sn and registry.get('machines', machine_sn).get('stolen'):
        registry.update('machines', machine_sn, revealed=int(time.time()))
        raise RuntimeError(_('Cannot register stolen "%s"') % machine_sn)

    migrate_uid = None
    if machine_sn:
        machine = registry.get('machines', machine_sn)
        if machine.get('uid') and machine['uid'] != uid:
            _logger.info(_('Found %s exiting registeration for %s, ' \
                    'will reuse its backup'), machine['uid'], machine_sn)
            migrate_uid = machine['uid']

    inserted = registry.update('users', uid, machine_sn=machine_sn,
            nickname=nickname, pubkey=pubkey,
            pending_restore=exists(env.hashed_backup_path(migrate_uid or uid)))
    if inserted:
        _logger.info('Registered new student')
    else:
        _logger.info('Changed registeration for existed student')

    if machine_sn:
        if migrate_uid and _move_backup(migrate_uid, uid):
            inserted = False
        registry.update('machines', machine_sn, uid=uid,
                machine_uuid=machine_uuid, registered=int(time.time()))

    if inserted and exists(_authorized_keys_path):
        keys_file = file(_authorized_keys_path, 'a')
        _add_authorized_key(keys_file, uid, pubkey)
        keys_file.close()
    else:
        # registeration was overriden, just recreate authorized_keys
        # instead of changing its content correspondingly
        create_keys()


def create_keys():
    _logger.debug('Create authorized_keys file')
    keys_file = util.new_file(_authorized_keys_path, 0600)
    for user in registry.find('users'):
        if not user.get('machine_sn') or \
                not registry.get('machines', user['machine_sn']).get('stolen'):
            _add_authorized_key(keys_file, user['uid'], user['pubkey'])
    keys_file.close()


def _move_backup(src_uid, dst_uid):
    src_backup = env.hashed_backup_path(src_uid)
    if not exists(src_backup):
        return False

    dst_backup = env.hashed_backup_path(dst_uid)
    if exists(dst_backup):
        _logger.info(_('Found existing backup for %s, abort moving'),
                dst_uid)
        return False

    _logger.info(_('Moving %s backup to %s'), src_uid, dst_uid)

    if not exists(dirname(dst_backup)):
        os.makedirs(dirname(dst_backup))
    os.rename(src_backup, dst_backup)

    return True


def _add_authorized_key(fd, uid, pubkey):
    machine_sn = registry.get('users', uid).get('machine_sn')
    if machine_sn and registry.get('machines', machine_sn).get('stolen'):
        return

    fd.write('command="%s %s",no-port-forwarding,' \
             'no-X11-forwarding,no-agent-forwarding,no-pty %s\n' % \
             (env.backup_path('sugar-server-rsync'), uid, pubkey))


def _registry_mtime():
    return max(registry.mtime('users'), registry.mtime('machines'))
