# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import pwd
import time
import hashlib
from SimpleXMLRPCServer import SimpleXMLRPCServer
from os.path import join, abspath
from gettext import gettext as _

from sugar_server import env, misc, registry, util
from sugar_server.util import enforce


etc = env.import_from('etc')
directory = env.import_from('directory')

_logger = env.get_logger('registry')
_compat_service = misc.ThreadingTCPServer()
_client_config_value = None


def setup():
    directory.setup()


def start():
    # Support OLPC XS's registration method
    compat_server = SimpleXMLRPCServer((env.hostname.value, 8080),
            logRequests=False)
    compat_server.register_function(_compat_register, 'register')
    _compat_service.start(compat_server)


def stop():
    _compat_service.stop()
    directory.close()


def GET_client_status(query):
    enforce('uid' in query,
            _('Query parameter "uid" needs to be passed with the request'))
    user = registry.get('users', query['uid'])
    if user:
        return {'registered': True,
                'backup-url': backup_url(),
                'jabber-url': etc.jabber_url.value,
                'pending-restore': user.get('pending_restore', False),
                }
    else:
        return {'registered': False,
                }


def POST_client_register(query, data):
    _logger.info(_('Got registration request: %r'), data)

    enforce(data.get('nickname', '').strip(), _('Nickname cannot be empty'))
    env.assert_pubkey(data.get('pubkey'))

    uid = _pubkey_to_uid(data['pubkey'])
    directory.register(uid, data['nickname'], data['pubkey'],
            data.get('machine_sn'), data.get('machine_uuid'))

    _logger.info('Student successfully registered')

    return {'uid': uid}


def backup_url():
    me = pwd.getpwuid(os.getuid())
    return '%s@%s:/' % (me.pw_name, env.hostname.value)


def import_root(root):
    users_root = join(abspath(root), 'home', 'registry', 'users')
    machines_root = join(abspath(root), 'home', 'registry', 'machines')
    items = registry.find(users_root)

    def next_item():
        item = next(items)
        machine = registry.get(machines_root, item.get('machine_sn'))
        return item, machine.get('machine_uuid')

    _import(next_item)


def import_xs(db_path):
    import sqlite3

    connection = sqlite3.connect(db_path)
    cursor = connection.cursor()
    cursor.execute('select * from laptops')

    def next_item():
        machine_sn, nickname, __, pubkey, uuid = cursor.next()[:5]
        return {'nickname': nickname,
                'pubkey': 'ssh-dss %s' % pubkey,
                'machine_sn': machine_sn,
                }, uuid

    _import(next_item)


def _import(next_item_cb):

    def import_item(item, machine_uuid):
        uid = _pubkey_to_uid(item['pubkey'])
        if registry.get('users', uid):
            _logger.debug('Skip importing %s, already registered',
                    item['machine_sn'])
            return False

        _logger.info(_('Import %s registry item'), item['machine_sn'])
        _logger.debug('Import registry item %r', item)

        item['imported'] = int(time.time())
        registry.update('users', uid, **item)

        machine = {'uid': uid}
        if machine_uuid:
            machine['machine_uuid'] = machine_uuid
        registry.update('machines', item['machine_sn'], **machine)

        return True

    updated = False
    try:
        item = None
        while True:
            try:
                item, machine_uuid = next_item_cb()
                updated = import_item(item, machine_uuid) or updated
            except StopIteration:
                break
            except Exception:
                util.exception(_logger, _('Failed to import %r'), item)
    finally:
        if updated:
            directory.create_keys()


def _compat_register(machine_sn, nickname, machine_uuid, pubkey):
    try:
        reply = POST_client_register({}, {
                'machine_sn': machine_sn,
                'nickname': nickname,
                'machine_uuid': machine_uuid,
                'pubkey': 'ssh-dss %s' % pubkey.strip(),
                })
        reply['success'] = 'OK'
        reply['backupurl'] = backup_url()
        reply['jabberserver'] = etc.jabber_url.value
        reply['backuppath'] = '/'
    except Exception, error:
        _logger.exception(_('Cannot process registration request'))
        reply = {'success': 'ERR', 'error': str(error)}
    return reply


def _pubkey_to_uid(pubkey):
    # The same algorithm that Sugar Shell uses to keep uids the same as JIDs
    return hashlib.sha1(pubkey.split()[1]).hexdigest()
