#!/usr/bin/env sugar-unit-client

# sugar-lint: disable

import os
import time
import shutil
from os.path import exists, dirname

from __init__ import IntegrationTest, main

from sugar_server import env


class ActivationIntegrationTest(IntegrationTest):

    def setUp(self):
        IntegrationTest.setUp(self)
        self.start(['registry', 'activation', 'keyring'])
        self.activation = env.import_from('service', 'activation')

    def test_activate_NotRegistered(self):
        self._import_lease('import')

        reply = self.bot.activate()

        self.assertEqual(file('import').read(), reply)

    def test_activate_ExistedLease(self):
        self._import_lease('import')

        self.bot.register()
        reply = self.bot.activate()

        self.assertEqual(file('import').read(), reply)

    def test_activate_NoLeaseAndDelegations(self):
        #self.bot.register()
        self.assertRaises(RuntimeError, self.bot.activate)

    def test_activate_DelegateLease(self):
        self._import_delegations()

        self.bot.register()
        reply = self.bot.activate()

        lease = self.activation.crypto.Lease(reply)
        self.assertEqual(self.bot.machine_sn, lease.machine_sn)
        assert not lease.expired

    def test_activate_PreferExisted(self):
        self._import_lease('import_lease')
        self._import_delegations()

        self.bot.register()
        reply = self.bot.activate()

        self.assertEqual(file('import_lease').read(), reply)

    def test_activate_ExistedExpired(self):
        self._import_lease('import', 0)
        time.sleep(1)

        self.bot.register()
        self.assertRaises(RuntimeError, self.bot.activate)

    def test_activate_FallbackToDelegationIfExistedExpired(self):
        self._import_lease('import_lease', 0)
        time.sleep(1)
        self._import_delegations()

        self.bot.register()
        reply = self.bot.activate()

        lease = self.activation.crypto.Lease(reply)
        self.assertEqual(self.bot.machine_sn, lease.machine_sn)
        assert not lease.expired

    def _import_lease(self, filename, days=30):
        env.call_crypto('make-lease', '--signingkey', self.master_lease, self.bot.machine_sn, self.bot.machine_uuid, days, filename)
        dst_path = self.activation.etc.leases_path(self.bot.machine_sn)
        if not exists(dirname(dst_path)):
            os.makedirs(dirname(dst_path))
        shutil.copy(filename, dst_path)

    def _import_delegations(self):
        self.touch(('import/delegations', '%s,%s' % (self.bot.machine_sn, self.bot.machine_uuid)))

        dst_path = self.activation.etc.lease_delegations_path(self.bot.machine_sn)
        if not exists(dirname(dst_path)):
            os.makedirs(dirname(dst_path))
        d_lease = env.call_crypto('make-server-delegations', 'import/delegations', 30, self.master_lease, env.root.value + '/home/keyring/private/server')
        self.touch((dst_path, d_lease))

        # TODO oats key doesn't affect on test results
        dst_path = self.activation.etc.oats_delegations_path(self.bot.machine_sn)
        if not exists(dirname(dst_path)):
            os.makedirs(dirname(dst_path))
        d_oats = env.call_crypto('make-server-delegations', 'import/delegations', 30, self.master_oats, env.root.value + '/home/keyring/private/server')
        self.touch((dst_path, d_oats))
