#!/usr/bin/env sugar-unit-client

# sugar-lint: disable

import os
import pwd
import time
from os.path import exists

from __init__ import IntegrationTest, main

from sugar_server import env, util, misc
from sugar_server.util import enforce


etc = env.import_from('etc', 'backup')


class BackupIntegrationTest(IntegrationTest):

    def setUp(self):
        IntegrationTest.setUp(self)

        etc.trim_timeout.value = 3600
        etc.trim_daily_limit.value = 365

        self.loadavg = 0
        self.override(os, 'getloadavg', lambda: [self.loadavg])

        self.ps = []
        self.override(misc, 'ps', lambda: self.ps)

        util._set_utcnow(60 * 60 * 24 * 0)

    def test_FreeSpaceLessThanHardQuote(self):
        etc.soft_quota.value = 100
        etc.hard_quota.value = 100
        self.start(['registry', 'backup'])
        env.set_statvfs(1, 1)

        self.bot.register()
        self.bot.store('foo', 'bar')
        self.assertRaises(RuntimeError, self.bot.backup)

        assert not exists(env.backup_path(self.bot.uid, 'foo'))

    def test_StartTrimmingAfterBackupAndDoingItOnlyOnceWithinTimout(self):
        etc.soft_quota.value = 50
        etc.hard_quota.value = 100
        etc.trim_timeout.value = 5
        self.start(['registry', 'backup'])
        env.set_statvfs(10, 4)

        self.bot.register()
        self.bot.store('foo')
        self.bot.backup()
        self.assertEqual(50, env.disk_usage())
        time.sleep(1)

        util._set_utcnow(60 * 60 * 24 * 1)
        self.bot.backup()
        self.assertEqual(60, env.disk_usage())
        time.sleep(1)

        util._set_utcnow(60 * 60 * 24 * 2)
        self.bot.backup()
        self.assertEqual(70, env.disk_usage())
        time.sleep(3)

        util._set_utcnow(60 * 60 * 24 * 3)
        self.bot.backup()
        self.assertEqual(50, env.disk_usage())
        time.sleep(1)

        util._set_utcnow(60 * 60 * 24 * 4)
        self.bot.backup()
        self.assertEqual(60, env.disk_usage())

    def test_NoTrimBeforeSoftQuota(self):
        etc.soft_quota.value = 50
        etc.hard_quota.value = 100
        self.start(['registry', 'backup'])
        env.set_statvfs(10, 0)

        self.bot.register()
        self.bot.store('foo')
        self.bot.backup()
        self.assertEqual(20, env.disk_usage())
        time.sleep(1)

        util._set_utcnow(60 * 60 * 24 * 1)
        self.bot.backup()
        self.assertEqual(30, env.disk_usage())
        time.sleep(1)

        util._set_utcnow(60 * 60 * 24 * 2)
        self.bot.backup()
        self.assertEqual(40, env.disk_usage())

    def test_EmergencyTrimAfterHardQuota(self):
        etc.soft_quota.value = 0
        etc.hard_quota.value = 30
        self.start(['registry', 'backup'])
        env.set_statvfs(10, 0)

        # Soft quota will play the first time
        self.bot.register()
        self.bot.store('foo')
        self.bot.backup()
        self.assertEqual(10, env.disk_usage())

        # Soft quota already played, not trimming
        self.bot.backup()
        self.assertEqual(20, env.disk_usage())

        # Next day, hard quota gained
        util._set_utcnow(60 * 60 * 24 * 1)
        self.bot.backup()
        self.assertEqual(10, env.disk_usage())

    def test_OnHardQuotaRejectNewBackupsAndRunEmergencyTrimming(self):
        etc.soft_quota.value = 0
        etc.hard_quota.value = 30
        self.start(['registry', 'backup'])
        env.set_statvfs(10, 0)

        self.bot.register()
        self.bot.store('foo')
        self.bot.backup()
        self.assertEqual(10, env.disk_usage())

        self.bot.backup()
        self.assertEqual(20, env.disk_usage())
        time.sleep(1)

        util._set_utcnow(60 * 60 * 24 * 1)
        env.set_statvfs(10, 1)
        self.assertRaises(RuntimeError, self.bot.backup)
        time.sleep(1)
        self.assertEqual(20, env.disk_usage())
