#!/usr/bin/env python

# sugar-lint: disable

import os
import pwd
import time
import shutil
from os.path import exists, join, dirname, abspath

from __init__ import UnitTest, main, Pwd

from sugar_server import env, util, misc, registry


users_path = join(dirname(abspath(__file__)), 'data', 'users')


class BackupTest(UnitTest):

    def setUp(self):
        UnitTest.setUp(self)
        self.setup_root()

        self.service = env.import_from('service', 'backup')
        self.etc = env.import_from('etc', 'backup')

        self.loadavg = 0
        self.override(os, 'getloadavg', lambda: [self.loadavg])

        self.disk_usage = 0
        self.override(env, 'disk_usage', lambda: self.disk_usage)

        self.ps = []
        self.override(misc, 'ps', lambda: self.ps)

        self.etc.hard_quota.value = 100
        self.etc.soft_quota.value = 100

    def test_start_TrimOnStartup(self):
        self.disk_usage = 100
        self.etc.soft_quota.value = 0
        self.etc.trim_timeout.value = 60

        try:
            self.service.start()
            time.sleep(1)
            assert exists('var/sugar-server/trim-backup.lock')
        finally:
            self.service.stop()

    def test_flow_opened(self):
        assert self.service.flow_opened()

        self.loadavg = 5
        assert self.service.flow_opened()
        self.loadavg = 6
        assert not self.service.flow_opened()
        self.loadavg = 0

        self.disk_usage = 49
        self.etc.hard_quota.value = 50
        assert self.service.flow_opened()

        self.etc.hard_quota.value = 49
        assert not self.service.flow_opened()
        self.etc.hard_quota.value = 100

        self.ps = [{'cmd': 'rsync'}] * 10
        assert self.service.flow_opened()
        self.ps = [{'cmd': 'rsync'}] * 11
        assert not self.service.flow_opened()
        self.ps = []

    def test_flow_opened_TrimOnDiskOverloaded(self):
        try:
            self.service.start()

            time.sleep(1)
            assert not exists('var/sugar-server/trim-backup.lock')

            self.disk_usage = 100
            self.etc.hard_quota.value = 0
            self.etc.soft_quota.value = 0
            assert not self.service.flow_opened()

            time.sleep(1)
            assert exists('var/sugar-server/trim-backup.lock')
        finally:
            self.service.stop()

    def test_GET_client_backup(self):
        registry.update('users', '8b442063dc1b104eb8ba1e6f00000018', machine_sn='SHF706002A7', pubkey='pubkey-1')
        os.makedirs('home/backup/18/8b442063dc1b104eb8ba1e6f00000018')
        self.assertEqual(
                {'accepted': True},
                self.service.GET_client_backup({'uid': '8b442063dc1b104eb8ba1e6f00000018'}))
        self.loadavg = 100
        self.assertEqual(
                {'accepted': False},
                self.service.GET_client_backup({'uid': '8b442063dc1b104eb8ba1e6f00000018'}))
        self.loadavg = 0

    def test_GET_client_backup_NotRegistered(self):
        self.assertRaises(RuntimeError, self.service.GET_client_backup, {'uid': '00000000000000000000000000000000'})

    def test_GET_client_backup_IncorrectQuery(self):
        registry.update('users', '8b442063dc1b104eb8ba1e6f00000018', machine_sn='SHF706002A7', pubkey='pubkey-1')
        os.makedirs('home/backup/18/8b442063dc1b104eb8ba1e6f00000018')
        self.assertRaises(RuntimeError, self.service.GET_client_backup, {})

    def test_import(self):

        def check():
            assert exists('home/backup/01/uid01/2011-09-21/1')
            assert exists('home/backup/01/uid01/2011-09-22/2')
            assert exists('home/backup/01/uid01/current/3')
            assert exists('home/backup/01/uid01/latest')
            self.assertEqual('2011-09-22', os.readlink('home/backup/01/uid01/latest'))
            assert not exists('home/backup/02')
            assert exists('home/backup/03/uid03/current/4')
            assert not exists('home/backup/03/uid03/latest')

        os.makedirs('LogVol00')
        shutil.copytree(users_path, 'LogVol00/users', symlinks=True)

        registry.update('machines', 'SHC020000B4', uid='uid01')
        registry.update('machines', 'SHC020000B9', uid='uid02')
        registry.update('machines', 'SHC05100431', uid='uid03')

        self.service.import_xs('LogVol00')
        check()

        os.makedirs('src')
        os.rename('home', 'src/home')
        os.makedirs('home/backup')

        self.service.import_root('src')
        check()


if __name__ == '__main__':
    main()
