#!/usr/bin/env python

# sugar-lint: disable

import datetime

from __init__ import UnitTest, main

from sugar_server import env, util, ipc, misc, registry, leases


class CryptoTest(UnitTest):

    def setUp(self):
        UnitTest.setUp(self)
        self.setup_root()
        self.crypto = env.import_from('crypto', 'activation')
        self.etc = env.import_from('etc', 'activation')
        self.etc.max_expiry.value = None
        self._keyring = misc.ThreadingTCPServer()

    def tearDown(self):
        self._keyring.stop()
        UnitTest.tearDown(self)

    def start_keyring(self, requests, replies):

        def handle(handler):
            args = {}
            command = args['command'] = handler.recv_message()
            if command == 'sign':
                args['expiry'] = str(leases.Expiry(handler.recv_message()))
            requests.append(args)
            assert replies
            handler.send_message(replies.pop(0) % args)

        self._keyring.start(ipc.Server(env.var_path('keyring'), handle))

    def test_get_ReturnExisted(self):
        requests = []
        replies = ['foo']
        self.start_keyring(requests, replies)

        lease_text = 'act01: SHF706002A7 K 90110529T135103Z sig01: sha256\n'
        self.touch(('share/leases/A7/SHF706002A7', lease_text))

        lease = self.crypto.get('SHF706002A7')
        assert not lease.delegated
        self.assertEqual('SHF706002A7', lease.machine_sn)
        self.assertEqual('90110529T135103Z', lease.expiry)
        assert replies

    def test_get_Delegate(self):
        self.etc.default_expiry.value = '70110529T135103Z'

        requests = []
        replies = ['act01: SHF706002A7 K 20110529T135103Z sig01: sha256']
        self.start_keyring(requests, replies)

        lease = self.crypto.get('SHF706002A7')
        self.assertEqual(
                [{'command': 'sign', 'expiry': '70110529T135103Z'}],
                requests)
        assert lease.delegated
        self.assertEqual('SHF706002A7', lease.machine_sn)
        assert not replies

    def test_get_IncorrectExistedLease(self):
        requests = []
        replies = ['act01: SHF706002A7 K 20110529T135103Z sig01: sha256']
        self.start_keyring(requests, replies)

        self.touch(('share/leases/A7/SHF706002A7', 'foo'))

        lease = self.crypto.get('SHF706002A7')
        self.assertEqual('SHF706002A7', lease.machine_sn)
        assert not replies

    def test_get_ExpiredExistedLease(self):
        requests = []
        replies = ['act01: SHF706002A7 K 20110529T135103Z sig01: sha256']
        self.start_keyring(requests, replies)

        lease_text = 'act01: SHF706002A7 K 00110529T135103Z sig01: sha256\n'
        self.touch(('share/leases/A7/SHF706002A7', lease_text))

        lease = self.crypto.get('SHF706002A7')
        self.assertEqual('SHF706002A7', lease.machine_sn)
        assert not replies

    def test_sign(self):
        requests = []
        replies = ['sig02: sha256 3082010a0282010100afc1cf127c727b6700ec4715e5e98620090588871bc3fd72f1bd96153be87954334f4b88959dd72e61ba0036d5f1d6d3194dca442ec0cff9785ea84e0a1c75921e84a5b1684731b76ae2394b5d39291ad6e4373ad1a54a102f08b77a73c792bb8cdcaf27e425583b2eb64ef0b525570af622acb2a5cd33e3ef410952100a4494de57b177dc4b5db6b312416a5d644d6a22273fe9cfd2ae972b61d49b644b74748153564e7f551b4f77f1802a1451adf73bbee049e6b378e893c3124b0ecacb927aff17fdba4ea8f7d828cb8913ac655e79bc624d67d78b88966962a67a207ac48a4494a7f38cdee4a786afb8f41c2b155b1d7d0567dd5225d92f46500c6a8f910203010001 20110529T135103Z 9e3efae0047c57666599a4ae7c39b98ad057598a2869f912edc810b7a8cabfbadfc583efe943f216b285036c0490c5349983356cfd16faacd0812a2a626104083340ef04726acf1c79b7632b6da6bd39c0d96614f8658b8637e41a0f96d9b5e844c273c741db81ac50196a42a675a38546e66a2e4a291d9af424570f53eee3724004ded21e6e5ce539ca5951f2db217c36b1cbc77ffb052f7d4770c4603c2001b030a6d923a06a25a8a770bf36631dc5f095fb6803168278576ed28e297cc938c65effcec12837baef41511d89129fab12d7388999b5c79304d5ceb701ae67ce44a9a2d101ea4679b3fc6fbd258ab452dfbc2b6e732e7c177544ae8e233d719c',
                  ]
        self.start_keyring(requests, replies)

        signed_orig = 'sig02: sha256 997068547fabe4c21c3f83c4e2fe77d8b14768a0812d311ae64eaf0203010001 20110529T191548Z 7375067ff39d239c445fc11e6d89683ed6b27c70f0f16f92a2937f49f05c80b9ec3db55a5a3d37e20802d3c2f4b46ec51396df01d658d96dfad1b9937a1a66e9bd6cd1349905984b92c3153485b3910d1938aa7b8985a764fd5b0fd7ddb02e89f11e17e7ef1224898ccd1b298c01f8679a805e74d4ae2ad280f342c645ffa2c289aaa99a168f355ace141ae213e5e54d3eec7e6e7d726299065062420b4913e581aa7a2e21f37ff1138939f60b9fb05923924e33d5186d1e3ec2f39b67aa64d862be825c808f17c126e0e3d44b48e807e896afc057706909890557c1fa4b88ce129a6538f10192f2dee6041f2e3279181e54415fd961ca5fbd54f487a4c7c203 sha256 3082010a0282010100afc1cf127c727b6700ec4715e5e98620090588871bc3fd72f1bd96153be87954334f4b88959dd72e61ba0036d5f1d6d3194dca442ec0cff9785ea84e0a1c75921e84a5b1684731b76ae2394b5d39291ad6e4373ad1a54a102f08b77a73c792bb8cdcaf27e425583b2eb64ef0b525570af622acb2a5cd33e3ef410952100a4494de57b177dc4b5db6b312416a5d644d6a22273fe9cfd2ae972b61d49b644b74748153564e7f551b4f77f1802a1451adf73bbee049e6b378e893c3124b0ecacb927aff17fdba4ea8f7d828cb8913ac655e79bc624d67d78b88966962a67a207ac48a4494a7f38cdee4a786afb8f41c2b155b1d7d0567dd5225d92f46500c6a8f910203010001 20110529T135103Z 9eff7c01cb46a45bcc5e1b5fc729e445eac2a8abe527d89fb5de35a9ca2c9d5c12db789d5ab577a9c3ecbe09af4b4990221f70c8f1bfc3bb7f0f2b83d127eb744fee56f318c4d265f6b31f13ce34972764168c336671f61c696cd8ab8fcc9f2d6b76437b4de803dead6111ae176b6dc2c2a40f8248efe4fd2f56d35b3c27720413280a6c808be03cac075ca8ebbcce86cade832832776ab3d52d1e8691877bdea99dc53ec2c8d11d87e35de30639e173f57b7d41f8d032030919f392ebb1bd2dae672f430b8cf237f5bd33b9fb1cfbded0aab10fe649a487b64ed3d7808d7d7e7748285134f1eb3b23fd458410f4309c63d87edededed227c2b2dee067c84fa8'
        signed = self.crypto.sign('SHF706002A7', '20110529T135103Z', 'foo')

        self.assertEqual(
                [{'command': 'sign', 'expiry': '20110529T135103Z'}],
                requests)
        self.assertEqual(' '.join(signed_orig.split()[:-1]), ' '.join(signed.split()[:-1]))
        self.verify('SHF706002A7:20110529T135103Z:foo', signed_orig.split()[-1])
        self.verify('SHF706002A7:20110529T135103Z:foo', signed.split()[-1])

    def test_sign_time(self):
        requests = []
        replies = ['sig02: sha256 3082010a0282010100afc1cf127c727b6700ec4715e5e98620090588871bc3fd72f1bd96153be87954334f4b88959dd72e61ba0036d5f1d6d3194dca442ec0cff9785ea84e0a1c75921e84a5b1684731b76ae2394b5d39291ad6e4373ad1a54a102f08b77a73c792bb8cdcaf27e425583b2eb64ef0b525570af622acb2a5cd33e3ef410952100a4494de57b177dc4b5db6b312416a5d644d6a22273fe9cfd2ae972b61d49b644b74748153564e7f551b4f77f1802a1451adf73bbee049e6b378e893c3124b0ecacb927aff17fdba4ea8f7d828cb8913ac655e79bc624d67d78b88966962a67a207ac48a4494a7f38cdee4a786afb8f41c2b155b1d7d0567dd5225d92f46500c6a8f910203010001 20110529T135103Z 4f6987a97272fd6b5ad0dc4d82e3d8ea27b0c2d9ba6a6ff2f3ed2dc59213adb1b80c53d508e1c93e46d6b5538d65c835bf489f5bd962e6a8dea015a6c9180f074a1d83f31814ae543f1f98e2e1b55ca699d9c3c2561f54ca57c3f74404fadea094c836862197b21f40095d958fa5547253bc07cd9a2d871d98cec245d7bca416e038a365ff60c6ef7afcd95e2eb8a84da177fe92d6ae9b89c718b0d7015cc0207c2f3a52c11924ab7945790b522b8fd78f58e22d7cedcda968500cf77fbb1f14e13c656148ccf3f374a825a511b041e47897c0eac4e0f9b1f4d65a23912c255170b48236310d9c7b7f413bc82f8a4f1f02f2a766399769d5b1572825a96f80dd',
                  ]
        self.start_keyring(requests, replies)

        signed_orig = 'time01: 00110529T135103Z sig02: sha256 997068547fabe4c21c3f83c4e2fe77d8b14768a0812d311ae64eaf0203010001 20110529T191548Z 7375067ff39d239c445fc11e6d89683ed6b27c70f0f16f92a2937f49f05c80b9ec3db55a5a3d37e20802d3c2f4b46ec51396df01d658d96dfad1b9937a1a66e9bd6cd1349905984b92c3153485b3910d1938aa7b8985a764fd5b0fd7ddb02e89f11e17e7ef1224898ccd1b298c01f8679a805e74d4ae2ad280f342c645ffa2c289aaa99a168f355ace141ae213e5e54d3eec7e6e7d726299065062420b4913e581aa7a2e21f37ff1138939f60b9fb05923924e33d5186d1e3ec2f39b67aa64d862be825c808f17c126e0e3d44b48e807e896afc057706909890557c1fa4b88ce129a6538f10192f2dee6041f2e3279181e54415fd961ca5fbd54f487a4c7c203 sha256 3082010a0282010100afc1cf127c727b6700ec4715e5e98620090588871bc3fd72f1bd96153be87954334f4b88959dd72e61ba0036d5f1d6d3194dca442ec0cff9785ea84e0a1c75921e84a5b1684731b76ae2394b5d39291ad6e4373ad1a54a102f08b77a73c792bb8cdcaf27e425583b2eb64ef0b525570af622acb2a5cd33e3ef410952100a4494de57b177dc4b5db6b312416a5d644d6a22273fe9cfd2ae972b61d49b644b74748153564e7f551b4f77f1802a1451adf73bbee049e6b378e893c3124b0ecacb927aff17fdba4ea8f7d828cb8913ac655e79bc624d67d78b88966962a67a207ac48a4494a7f38cdee4a786afb8f41c2b155b1d7d0567dd5225d92f46500c6a8f910203010001 20110529T135103Z 4fcdb52bbfa194c161f54df17bcb73ce1a07fba1ca1c3560035318e3888dd3498fdd097b8629ea7158ef6d2661899b251f1323943f04b368256dae7224af4ddd55502f5b3808f7588006630935514c2a6d16851ec5315bdeedfbf956d3cb171a165055fe0da499959e7b58ac6c5f0d375657149393e26e9313ed33f7acee9b692281b8c10c04c07c88692152f8da9bdf6b99e43ca4693109fb9b2f7d603b5fd7db9e0cb122937682ba1a9367cab318d407ca6f9491d4c432f5a3e15e7552e5fe7dbf4eeec0598e79de70fed6fdc5cf45a4315a81e118e3f14818490acb4a7500d11d8344ccc928084d0e1c8274f8560414f8dd3f1ff98f485e6a23af4320b6c0'
        signed = self.crypto.sign_time('SHF706002A7', 'foo', '20110529T135103Z')

        self.assertEqual(
                [{'command': 'sign', 'expiry': '20110529T135103Z'}],
                requests)
        self.assertEqual(' '.join(signed_orig.split()[:-1]), ' '.join(signed.split()[:-1]))
        self.verify('SHF706002A7:20110529T135103Z:00110529T135103Z:foo', signed_orig.split()[-1])
        self.verify('SHF706002A7:20110529T135103Z:00110529T135103Z:foo', signed.split()[-1])

    def test_get_Stolen(self):
        self.assertEqual(
                '48621d9d1e11f66ad8cba8a7bb32a1998fd9a55d5c9256661aef48f22f56a9e9',
                self.crypto.get_stolen_status('uuid-1', False, 'foo'))
        self.assertEqual(
                'a2f92244f08029d504050bd454067113c63f1ccda3dd985dc8cd57761a1b29af',
                self.crypto.get_stolen_status('uuid-1', True, 'foo'))

    def test_delegate(self):
        self.start_real_keyring()

        lease_orig = 'act01: SHF706002A7 K 20110529T135103Z sig02: sha256 997068547fabe4c21c3f83c4e2fe77d8b14768a0812d311ae64eaf0203010001 20110529T191548Z 7375067ff39d239c445fc11e6d89683ed6b27c70f0f16f92a2937f49f05c80b9ec3db55a5a3d37e20802d3c2f4b46ec51396df01d658d96dfad1b9937a1a66e9bd6cd1349905984b92c3153485b3910d1938aa7b8985a764fd5b0fd7ddb02e89f11e17e7ef1224898ccd1b298c01f8679a805e74d4ae2ad280f342c645ffa2c289aaa99a168f355ace141ae213e5e54d3eec7e6e7d726299065062420b4913e581aa7a2e21f37ff1138939f60b9fb05923924e33d5186d1e3ec2f39b67aa64d862be825c808f17c126e0e3d44b48e807e896afc057706909890557c1fa4b88ce129a6538f10192f2dee6041f2e3279181e54415fd961ca5fbd54f487a4c7c203 sha256 3082010a0282010100afc1cf127c727b6700ec4715e5e98620090588871bc3fd72f1bd96153be87954334f4b88959dd72e61ba0036d5f1d6d3194dca442ec0cff9785ea84e0a1c75921e84a5b1684731b76ae2394b5d39291ad6e4373ad1a54a102f08b77a73c792bb8cdcaf27e425583b2eb64ef0b525570af622acb2a5cd33e3ef410952100a4494de57b177dc4b5db6b312416a5d644d6a22273fe9cfd2ae972b61d49b644b74748153564e7f551b4f77f1802a1451adf73bbee049e6b378e893c3124b0ecacb927aff17fdba4ea8f7d828cb8913ac655e79bc624d67d78b88966962a67a207ac48a4494a7f38cdee4a786afb8f41c2b155b1d7d0567dd5225d92f46500c6a8f910203010001 20110529T135103Z 354153b6300a97b58d31d97791d96cef902c8d6444289ca03d53c10ed4eb9d1addf3af8a02f5981e99b8d38b626e641f0fe948cfa9761a72b3c9b58d69befe4696b805a37f080090c8a0519f8b97489e2c2fdb6dac1ac58b8d59fabd755de6e18ed6b51bab97d90272d44fd758b40884261e499021eee82521977b6cebf47a38121c59d5d6904e8a8e2b32682bf422264c6f2c4183d07288987c0c84acd8dbd2a03d724efc6232fe4e64fbb2ef17a1043b95ee053f3bd3a0263843cee5265db70da51af4ef39edd7a51f97b0e17fc25366c50ac1825aa08862954ffe7520df3e632862783b22cc325df039721c670de4107727d6e0aeb95364f8b2ca1761a8f6'
        lease = self.crypto.delegate('SHF706002A7', '20110529T135103Z')

        self.assertEqual(' '.join(lease_orig.split()[:-1]), ' '.join(lease.split()[:-1]))
        self.verify('SHF706002A7:20110529T135103Z:SHF706002A7:00000000000000000000000000000001:K:20110529T135103Z', lease_orig.split()[-1])
        self.verify('SHF706002A7:20110529T135103Z:SHF706002A7:00000000000000000000000000000001:K:20110529T135103Z', lease.split()[-1])

    def test_delegate_WrongSerial(self):
        self.touch((
            'share/lease-delegations/A7/SHF706002A7',
            'del01: SHF70000000 00000000000000000000000000000001 sig02: sha256 997068547fabe4c21c3f83c4e2fe77d8b14768a0812d311ae64eaf0203010001 20110529T191548Z 7375067ff39d239c445fc11e6d89683ed6b27c70f0f16f92a2937f49f05c80b9ec3db55a5a3d37e20802d3c2f4b46ec51396df01d658d96dfad1b9937a1a66e9bd6cd1349905984b92c3153485b3910d1938aa7b8985a764fd5b0fd7ddb02e89f11e17e7ef1224898ccd1b298c01f8679a805e74d4ae2ad280f342c645ffa2c289aaa99a168f355ace141ae213e5e54d3eec7e6e7d726299065062420b4913e581aa7a2e21f37ff1138939f60b9fb05923924e33d5186d1e3ec2f39b67aa64d862be825c808f17c126e0e3d44b48e807e896afc057706909890557c1fa4b88ce129a6538f10192f2dee6041f2e3279181e54415fd961ca5fbd54f487a4c7c203',
            ))
        self.assertRaises(RuntimeError, self.crypto.delegate, 'SHF706002A7', '20110529T135103Z')

    def test_delegate_RestictByMaximum(self):
        self.start_real_keyring()
        self.etc.max_expiry.value = '20110529T135103Z'
        lease = self.crypto.delegate('SHF706002A7', '90110529T135103Z')
        assert lease.startswith('act01: SHF706002A7 K 20110529T135103Z')


if __name__ == '__main__':
    main()
