#!/usr/bin/env python

# sugar-lint: disable

import os
import time
from os.path import exists, isdir

from __init__ import UnitTest, main

from sugar_server import env, registry


class DirectoryTest(UnitTest):

    def setUp(self):
        UnitTest.setUp(self)
        self.setup_root()
        self.directory = env.import_from('service', 'registry').directory
        self.directory.setup()

    def test_register_assertion(self):
        self.directory.register('uid', 'nickname-1', 'pubkey-1', 'AAA00000001', '00000000000000000000000000000001')
        self.assertRaises(RuntimeError, self.directory.register, 'uid', 'nickname-1\n', 'pubkey-1', 'AAA00000001', '00000000000000000000000000000001')
        self.assertRaises(RuntimeError, self.directory.register, 'uid', 'nickname-1', 'pubkey-1\n', 'AAA00000001', '00000000000000000000000000000001')

    def test_register(self):
        self.directory.register('uid', 'nickname-1', 'pubkey-1', 'AAA00000001', '00000000000000000000000000000001')
        self.assertEqual(
                0600,
                os.stat('.ssh/authorized_keys').st_mode & 0777)
        self.assertEqual(
                ['command="./home/backup/sugar-server-rsync uid",no-port-forwarding,no-X11-forwarding,no-agent-forwarding,no-pty pubkey-1\n'],
                file('.ssh/authorized_keys').readlines())

        self.directory.register('uid-2', 'nickname-2', 'pubkey-2', 'AAA00000002', '00000000000000000000000000000002')
        self.assertEqual(
                0600,
                os.stat('.ssh/authorized_keys').st_mode & 0777)
        self.assertEqual(
                ['command="./home/backup/sugar-server-rsync uid",no-port-forwarding,no-X11-forwarding,no-agent-forwarding,no-pty pubkey-1\n',
                 'command="./home/backup/sugar-server-rsync uid-2",no-port-forwarding,no-X11-forwarding,no-agent-forwarding,no-pty pubkey-2\n'],
                file('.ssh/authorized_keys').readlines())

    def test_register_NoStolen(self):
        registry.update('machines', 'AAA00000001', stolen=False)
        self.directory.register('uid', 'nickname-1', 'pubkey-1', 'AAA00000001', '00000000000000000000000000000001')
        assert 'revealed' not in registry.get('machines', 'AAA00000001')
        registry.update('machines', 'AAA00000002', stolen=True)
        self.assertRaises(RuntimeError, self.directory.register, 'uid-2', 'nickname-2', 'pubkey-2', 'AAA00000002', '00000000000000000000000000000002')
        assert 'revealed' in registry.get('machines', 'AAA00000002')

    def test_setup_NoStolen(self):
        registry.update('users', 'uid-1', pubkey='pubkey-1', machine_sn='AAA00000001')
        registry.update('machines', 'AAA00000001', stolen=True)
        registry.update('users', 'uid-2', pubkey='pubkey-2', machine_sn='AAA00000002')
        registry.update('machines', 'AAA00000002', stolen=False)

        os.unlink('.ssh/authorized_keys')
        self.directory.setup()

        self.assertEqual(
                ['command="./home/backup/sugar-server-rsync uid-2",no-port-forwarding,no-X11-forwarding,no-agent-forwarding,no-pty pubkey-2\n'],
                file('.ssh/authorized_keys').readlines())

    def test_setup_RefreshAuthorizedKeys(self):
        os.utime('.ssh/authorized_keys', (0, 0))
        self.directory.setup()
        assert os.stat('.ssh/authorized_keys').st_mtime == 0

        os.makedirs('home/registry/users')
        self.directory.setup()
        assert os.stat('.ssh/authorized_keys').st_mtime != 0

        os.utime('.ssh/authorized_keys', (0, 0))
        os.makedirs('home/registry/machines')
        self.directory.setup()
        assert os.stat('.ssh/authorized_keys').st_mtime != 0

    def test_register_Reregister(self):
        self.directory.register('uid-1', 'nickname-1', 'pubkey-1', 'AAA00000001', '00000000000000000000000000000001')
        old_ctime = os.stat('.ssh').st_ctime

        time.sleep(1)

        self.directory.register('uid-2', 'nickname-2', 'pubkey-2', 'AAA00000002', '00000000000000000000000000000002')
        new_ctime = os.stat('.ssh').st_ctime
        assert old_ctime == new_ctime

        self.directory.register('uid-1', 'nickname-1', 'pubkey-1', 'AAA00000001', '00000000000000000000000000000001')
        new_ctime = os.stat('.ssh').st_ctime
        assert old_ctime < new_ctime
        self.assertEqual(
                0600,
                os.stat('.ssh/authorized_keys').st_mode & 0777)
        self.assertEqual(
                sorted([
                    'command="./home/backup/sugar-server-rsync uid-1",no-port-forwarding,no-X11-forwarding,no-agent-forwarding,no-pty pubkey-1\n',
                    'command="./home/backup/sugar-server-rsync uid-2",no-port-forwarding,no-X11-forwarding,no-agent-forwarding,no-pty pubkey-2\n',
                    ]),
                sorted(file('.ssh/authorized_keys').readlines()))

    def test_register_MoveBackupFromExistinSN(self):
        self.directory.register('uid01', 'nickname-1', 'pubkey-1', 'AAA00000001', '00000000000000000000000000000001')
        self.touch('home/backup/01/uid01/test')
        self.directory.register('uid02', 'nickname-2', 'pubkey-2', 'AAA00000001', '00000000000000000000000000000001')

        assert exists('home/backup/02/uid02/test')
        assert not exists('home/backup/01/uid01')

        self.assertEqual(
                sorted([
                    'command="./home/backup/sugar-server-rsync uid01",no-port-forwarding,no-X11-forwarding,no-agent-forwarding,no-pty pubkey-1\n',
                    'command="./home/backup/sugar-server-rsync uid02",no-port-forwarding,no-X11-forwarding,no-agent-forwarding,no-pty pubkey-2\n',
                    ]),
                sorted(file('.ssh/authorized_keys').readlines()))

        self.assertEqual('uid02', registry.get('machines', 'AAA00000001').get('uid'))
        self.assertEqual('AAA00000001', registry.get('users', 'uid01').get('machine_sn'))
        self.assertEqual('AAA00000001', registry.get('users', 'uid02').get('machine_sn'))

    def test_register_SetPendingRestore(self):
        self.directory.register('uid', 'nickname', 'pubkey', 'AAA00000001', '00000000000000000000000000000001')
        assert not registry.get('users', 'uid')['pending_restore']

        os.makedirs(env.hashed_backup_path('uid'))
        self.directory.register('uid', 'nickname', 'pubkey', 'AAA00000001', '00000000000000000000000000000001')
        assert registry.get('users', 'uid')['pending_restore']

        self.directory.register('uid2', 'nickname', 'pubkey', 'AAA00000001', '00000000000000000000000000000001')
        assert registry.get('users', 'uid2')['pending_restore']


if __name__ == '__main__':
    main()
