#!/usr/bin/env python

# sugar-lint: disable

import os
import pwd
import time
import hashlib
from os.path import join, dirname, abspath

from __init__ import UnitTest, main

from sugar_server import util, env, misc, registry


db_path = join(dirname(abspath(__file__)), 'data', 'identity.db')


class RegistryTest(UnitTest):

    def setUp(self):
        UnitTest.setUp(self)
        self.setup_root()

        self.service = env.import_from('service', 'registry')
        self.service.setup()
        self.etc = env.import_from('etc', 'registry')

    def test_Register_Asserts(self):
        self.assertRaises(RuntimeError, self.service.POST_client_register, {}, {
            'machine_sn': 'AAA00000001',
            'pubkey': 'ssh-rsa pubkey',
            })
        self.assertRaises(RuntimeError, self.service.POST_client_register, {}, {
            'nickname': '',
            'machine_sn': 'AAA00000001',
            'pubkey': 'ssh-rsa pubkey',
            })
        self.assertRaises(RuntimeError, self.service.POST_client_register, {}, {
            'nickname': 'nick',
            'machine_sn': 'BAD',
            'pubkey': 'ssh-rsa pubkey',
            })
        self.assertRaises(RuntimeError, self.service.POST_client_register, {}, {
            'nickname': 'nick',
            'machine_sn': 'AAA00000001',
            'pubkey': 'pubkey',
            })
        self.service.POST_client_register({}, {
            'nickname': 'nick',
            'machine_sn': 'AAA00000001',
            'pubkey': 'ssh-rsa pubkey',
            })

    def test_Register(self):
        uid = hashlib.sha1('pubkey').hexdigest()

        self.assertEqual(
                {'uid': uid},
                self.service.POST_client_register({}, {
                    'nickname': 'nick',
                    'machine_sn': 'AAA00000001',
                    'pubkey': 'ssh-rsa pubkey',
                    }))

        assert registry.get('users', uid)
        assert registry.get('machines', 'AAA00000001')['registered']

    def test_Register_SetPendingRestore(self):
        uid = hashlib.sha1('pubkey').hexdigest()
        os.makedirs(env.hashed_backup_path(uid))
        self.service.POST_client_register({}, {
            'nickname': 'nick',
            'machine_sn': 'AAA00000001',
            'pubkey': 'ssh-rsa pubkey',
            })
        self.assertEqual(
                True,
                self.service.GET_client_status({'uid': uid}).get('pending-restore'))

    def test_RegisterCompat(self):
        reply = self.service._compat_register('bad', 'nickname-1', '00000000000000000000000000000001', 'pubkey-1')
        self.assertEqual(
                {'success': 'ERR',
                 'error': 'Invalid machine serial number: bad'},
                reply)
        assert not [i for i in registry.find('users')]

        reply = self.service._compat_register('AAA00000001', 'nickname-1', '00000000000000000000000000000001', 'pubkey-1')
        self.assertEqual('OK', reply.get('success'))
        self.assertEqual(self.service.backup_url(), reply.get('backupurl'))
        self.assertEqual(self.etc.jabber_url.value, reply.get('jabberserver'))
        self.assertEqual('/', reply.get('backuppath'))
        uid = hashlib.sha1('pubkey-1').hexdigest()
        assert registry.get('users', uid)
        assert registry.get('machines', 'AAA00000001')['registered']

    def test_GET_client_status(self):
        self.assertEqual(
                {'registered': False},
                self.service.GET_client_status({'uid': '00000000000000000000000000000001'}))
        reply = self.service.POST_client_register({}, {
            'pubkey': 'ssh-rsa pubkey-1',
            'nickname': 'nickname-1',
            })
        self.assertEqual(
                {'registered': True,
                 'jabber-url': self.etc.jabber_url.value,
                 'backup-url': self.service.backup_url(),
                 'pending-restore': False,
                 },
                self.service.GET_client_status({'uid': reply['uid']}))

    def test_import_xs(self):
        self.override(time, 'time', lambda: 0)
        self.service.import_xs(db_path)

        users = []
        authorized_keys = []
        for i in range(7):
            users.append({
                'machine_sn': 'AAA0000000%s' % (i + 1),
                'nickname': 'user-%s' % (i + 1),
                'pubkey': 'ssh-dss pubkey-%s' % (i + 1),
                'uid': hashlib.sha1('pubkey-%s' % (i + 1)).hexdigest(),
                'imported': 0,
                })
            authorized_keys.append(
                    'command="./home/backup/sugar-server-rsync %s",no-port-forwarding,no-X11-forwarding,no-agent-forwarding,no-pty %s\n' % \
                            (hashlib.sha1('pubkey-%s' % (i + 1)).hexdigest(), 'ssh-dss pubkey-%s' % (i + 1)))
        self.assertEqual(
                sorted(users),
                sorted([i for i in registry.find('users')]))
        self.assertEqual(
                sorted(authorized_keys),
                sorted(file('.ssh/authorized_keys').readlines()))

        machines = []
        for i in range(7):
            machines.append({
                'machine_sn': 'AAA0000000%s' % (i + 1),
                'machine_uuid': 'uuid-%s' % (i + 1),
                'uid': hashlib.sha1('pubkey-%s' % (i + 1)).hexdigest(),
                })
        self.assertEqual(
                sorted(machines),
                sorted([i for i in registry.find('machines')]))

    def test_import_root(self):
        self.override(time, 'time', lambda: 0)
        self.service.import_xs(db_path)

        self.override(time, 'time', lambda: 1)
        os.makedirs('src')
        os.rename('home', 'src/home')
        self.service.import_root('src')

        users = []
        authorized_keys = []
        for i in range(7):
            users.append({
                'machine_sn': 'AAA0000000%s' % (i + 1),
                'nickname': 'user-%s' % (i + 1),
                'pubkey': 'ssh-dss pubkey-%s' % (i + 1),
                'uid': hashlib.sha1('pubkey-%s' % (i + 1)).hexdigest(),
                'imported': 1,
                })
            authorized_keys.append(
                    'command="./home/backup/sugar-server-rsync %s",no-port-forwarding,no-X11-forwarding,no-agent-forwarding,no-pty %s\n' % \
                            (hashlib.sha1('pubkey-%s' % (i + 1)).hexdigest(), 'ssh-dss pubkey-%s' % (i + 1)))
        self.assertEqual(
                sorted(users),
                sorted([i for i in registry.find('users')]))
        self.assertEqual(
                sorted(authorized_keys),
                sorted(file('.ssh/authorized_keys').readlines()))

        machines = []
        for i in range(7):
            machines.append({
                'machine_sn': 'AAA0000000%s' % (i + 1),
                'machine_uuid': 'uuid-%s' % (i + 1),
                'uid': hashlib.sha1('pubkey-%s' % (i + 1)).hexdigest(),
                })
        self.assertEqual(
                sorted(machines),
                sorted([i for i in registry.find('machines')]))


if __name__ == '__main__':
    main()
