#!/usr/bin/env python

# sugar-lint: disable

import os
import time
import datetime
from os.path import exists, getsize, islink, abspath

from __init__ import UnitTest, main

from sugar_server import env, util, misc


class RsyncTest(UnitTest):

    def setUp(self):
        UnitTest.setUp(self)
        self.rsync = env.import_from('rsync', 'backup')
        self.etc = env.import_from('etc', 'backup')

        util._set_utcnow(0)
        self.backup_dirname = '1970-01-01'

    def test_cook_rsync_command_SSH_ORIGINAL_COMMAND(self):
        if 'SSH_ORIGINAL_COMMAND' in os.environ:
            del os.environ['SSH_ORIGINAL_COMMAND']
        self.assertRaises(RuntimeError, self.rsync.cook_rsync_command, '.')

        os.environ['SSH_ORIGINAL_COMMAND'] = 'echo'
        self.assertRaises(RuntimeError, self.rsync.cook_rsync_command, '.')

    def test_cook_rsync_command_RequiredArguments(self):
        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync'
        self.assertRaises(RuntimeError, self.rsync.cook_rsync_command, '.')

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server'
        self.assertRaises(RuntimeError, self.rsync.cook_rsync_command, '.')

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server .'
        self.assertRaises(RuntimeError, self.rsync.cook_rsync_command, '.')

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server foo current'
        self.assertRaises(RuntimeError, self.rsync.cook_rsync_command, '.')

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . current'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/current/'],
                self.rsync.cook_rsync_command('.')[0])

    def test_cook_rsync_command_ReplaceWrongDst(self):
        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . foo'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/current/foo'],
                self.rsync.cook_rsync_command('.')[0])

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . /foo/bar'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/current/foo/bar'],
                self.rsync.cook_rsync_command('.')[0])

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . /foo/bar/'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/current/foo/bar/'],
                self.rsync.cook_rsync_command('.')[0])

    def test_cook_rsync_command_ParticularDate(self):
        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . /2011-01-01'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/2011-01-01/'],
                self.rsync.cook_rsync_command('.')[0])

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . /2011-01-01/'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/2011-01-01/'],
                self.rsync.cook_rsync_command('.')[0])

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . /2011-01-01/foo'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/2011-01-01/foo'],
                self.rsync.cook_rsync_command('.')[0])

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . /2011-01-01/foo/'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/2011-01-01/foo/'],
                self.rsync.cook_rsync_command('.')[0])

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . /2011-01-0a/'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/current/2011-01-0a/'],
                self.rsync.cook_rsync_command('.')[0])

    def test_cook_rsync_command_SupportDeprecatedDst(self):
        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . datastore-current/'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/current/'],
                self.rsync.cook_rsync_command('.')[0])

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . datastore-current/foo'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/current/foo'],
                self.rsync.cook_rsync_command('.')[0])

    def test_cook_rsync_command_DstLastSlash(self):
        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . current/'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/current/'],
                self.rsync.cook_rsync_command('.')[0])

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . current'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/current/'],
                self.rsync.cook_rsync_command('.')[0])

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . /'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/current/'],
                self.rsync.cook_rsync_command('.')[0])

    def test_cook_rsync_command_AccessOutOfPersonalDirectory(self):
        os.environ['SSH_ORIGINAL_COMMAND'] = \
                'rsync --server -e.iLsf . current/1/..'
        self.assertEqual(
                ['/usr/bin/rsync', '--server', '--rsh', '.iLsf', '.', abspath(os.curdir) + '/current/'],
                self.rsync.cook_rsync_command('.')[0])

        os.environ['SSH_ORIGINAL_COMMAND'] = \
                'rsync --server -e.iLsf . current/1/../..'
        self.assertRaises(RuntimeError, self.rsync.cook_rsync_command, '.')

        os.environ['SSH_ORIGINAL_COMMAND'] = \
                'rsync --server -e.iLsf . current/..'
        self.assertRaises(RuntimeError, self.rsync.cook_rsync_command, '.')

        os.environ['SSH_ORIGINAL_COMMAND'] = \
                'rsync --server -e.iLsf . current/../..'
        self.assertRaises(RuntimeError, self.rsync.cook_rsync_command, '.')

        os.environ['SSH_ORIGINAL_COMMAND'] = \
                'rsync --server -e.iLsf . ..'
        self.assertRaises(RuntimeError, self.rsync.cook_rsync_command, '.')

        os.environ['SSH_ORIGINAL_COMMAND'] = \
                'rsync --server -e.iLsf . /..'
        self.assertRaises(RuntimeError, self.rsync.cook_rsync_command, '.')

    def test_cook_rsync_command_DSBackupClientRequests(self):
        os.environ['SSH_ORIGINAL_COMMAND'] = \
                'rsync --server -ltrze.iLsf --timeout=160 --delete --partial . current'
        self.assertEqual(
                ['/usr/bin/rsync',
                 '--server', '--rsh', '.iLsf',
                 '--links', '--times', '--recursive', '--compress',
                 '--timeout=160', '--delete', '--partial',
                 '.', abspath(os.curdir) + '/current/',
                 ],
                self.rsync.cook_rsync_command('.')[0])

        os.environ['SSH_ORIGINAL_COMMAND'] = \
                'rsync --server -ltrze.iLsf --timeout=160 --delete --partial --temp-dir /tmp . /current'
        self.assertEqual(
                ['/usr/bin/rsync',
                 '--server', '--rsh', '.iLsf',
                 '--links', '--times', '--recursive', '--compress',
                 '--timeout=160', '--delete', '--partial',
                 '.', abspath(os.curdir) + '/current/'],
                self.rsync.cook_rsync_command('.')[0])

    def test_postprocess(self):
        self.setup_root()
        self.etc.soft_quota.value = 100
        root = 'home/backup/go/00000000001000000000200000000031/'
        self.touch(root + 'current/foo')

        self.rsync.postprocess(root)
        assert exists(root + self.backup_dirname)
        assert islink(root + 'latest')
        self.assertEqual(self.backup_dirname, os.readlink(root + 'latest'))

    def test_postprocessTrimJustBackedUp(self):
        self.setup_root()
        self.etc.soft_quota.value = 0
        root = 'home/backup/go/00000000001000000000200000000031/'
        self.touch(root + 'current/')

        self.rsync.postprocess(root)
        assert not exists(root + self.backup_dirname)
        assert islink(root + 'latest')
        self.assertEqual('current', os.readlink(root + 'latest'))

    def test_postprocess_Overwrites(self):
        self.setup_root()
        self.etc.soft_quota.value = 100
        root = 'home/backup/go/00000000001000000000200000000031/'
        self.touch(root + 'current/')

        self.touch(root + self.backup_dirname + '/foo')
        assert exists(root + self.backup_dirname + '/foo')
        os.symlink('.', root + 'latest')

        self.rsync.postprocess(root)
        assert not exists(root + self.backup_dirname + '/foo')
        assert exists(root + self.backup_dirname)
        self.assertEqual(self.backup_dirname, os.readlink(root + 'latest'))

    def test_postprocess_RemoveRemainsOnFails(self):
        self.setup_root()
        self.etc.soft_quota.value = 100
        root = 'home/backup/go/00000000001000000000200000000031/'
        self.touch(root + 'current/')
        os.symlink('.', root + 'latest')

        self.override(env, 'import_from', lambda *args: None)
        self.assertRaises(Exception, self.rsync.postprocess, root)

        assert not exists(root + self.backup_dirname)
        self.assertEqual('current', os.readlink(root + 'latest'))

    def test_cook_rsync_command_DstLastSlash(self):
        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server . /'
        self.assertEqual(
                (['/usr/bin/rsync', '--server', '.', abspath(os.curdir) + '/current/'], None),
                self.rsync.cook_rsync_command('.'))

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server -n . /'
        self.assertEqual(
                (['/usr/bin/rsync', '--server', '--dry-run', '.', abspath(os.curdir) + '/current/'], True),
                self.rsync.cook_rsync_command('.'))

        os.environ['SSH_ORIGINAL_COMMAND'] = 'rsync --server --dry-run . /'
        self.assertEqual(
                (['/usr/bin/rsync', '--server', '--dry-run', '.', abspath(os.curdir) + '/current/'], True),
                self.rsync.cook_rsync_command('.'))


if __name__ == '__main__':
    main()
