#!/usr/bin/env python

# sugar-lint: disable

import os
import time
from os.path import exists, getsize

from __init__ import UnitTest, main

from sugar_server import env, ipc, util


class TrimmerTest(UnitTest):

    def setUp(self):
        UnitTest.setUp(self)
        self.setup_root()

        self.trimmer = env.import_from('trimmer', 'backup')
        self.etc = env.import_from('etc', 'backup')

        self.etc.trim_daily_limit.value = 365
        self.etc.soft_quota.value = 0
        self.etc.hard_quota.value = 100

    def test_walk_EmptyDirectory(self):
        assert not [i for i in self.trimmer.walk('home/backup')]

        os.makedirs('home/backup/foo')
        assert not [i for i in self.trimmer.walk('home/backup')]

    def test_walk_OnlyDirectories(self):
        backup_filename = 'backup/2011-05-31'
        self.touch(backup_filename)
        assert not [i for i in self.trimmer.walk('backup')]
        os.unlink(backup_filename)
        os.makedirs(backup_filename)
        self.assertEqual(1, len([i for i in self.trimmer.walk('backup')]))

    def test_walk(self):
        dirs = ['backup/2011-12-30',
                'backup/2011-12-15',
                'backup/2011-12-05',

                'backup/2011-11-30',
                'backup/2011-11-05',

                'backup/2011-10-01',
                ]
        for i in dirs:
            os.makedirs(i)

        self.etc.trim_daily_limit.value = 0
        self.assertEqual([
                'backup/2011-11-05',

                'backup/2011-12-05',
                'backup/2011-12-15',

                'backup/2011-10-01',
                'backup/2011-11-30',
                'backup/2011-12-30',
                ],
                [i for i in self.trimmer.walk('backup')])

        self.etc.trim_daily_limit.value = 1
        self.assertEqual([
                'backup/2011-11-05',

                'backup/2011-12-05',
                'backup/2011-12-15',

                'backup/2011-10-01',
                'backup/2011-11-30',
                'backup/2011-12-30',
                ],
                [i for i in self.trimmer.walk('backup')])

        self.etc.trim_daily_limit.value = 15
        self.assertEqual([
                'backup/2011-11-05',

                'backup/2011-12-05',
                'backup/2011-12-15',

                'backup/2011-10-01',
                'backup/2011-11-30',
                'backup/2011-12-30',
                ],
                [i for i in self.trimmer.walk('backup')])

        self.etc.trim_daily_limit.value = 16
        self.assertEqual([
                'backup/2011-11-05',

                'backup/2011-12-05',

                'backup/2011-10-01',
                'backup/2011-11-30',
                'backup/2011-12-15',
                'backup/2011-12-30',
                ],
                [i for i in self.trimmer.walk('backup')])

        self.etc.trim_daily_limit.value = 25
        self.assertEqual([
                'backup/2011-11-05',

                'backup/2011-12-05',

                'backup/2011-10-01',
                'backup/2011-11-30',
                'backup/2011-12-15',
                'backup/2011-12-30',
                ],
                [i for i in self.trimmer.walk('backup')])

        self.etc.trim_daily_limit.value = 26
        self.assertEqual([
                'backup/2011-11-05',

                'backup/2011-10-01',
                'backup/2011-11-30',
                'backup/2011-12-05',
                'backup/2011-12-15',
                'backup/2011-12-30',
                ],
                [i for i in self.trimmer.walk('backup')])

        self.etc.trim_daily_limit.value = 30
        self.assertEqual([
                'backup/2011-11-05',

                'backup/2011-10-01',
                'backup/2011-11-30',
                'backup/2011-12-05',
                'backup/2011-12-15',
                'backup/2011-12-30',
                ],
                [i for i in self.trimmer.walk('backup')])

        self.etc.trim_daily_limit.value = 31
        self.assertEqual([
                'backup/2011-11-05',

                'backup/2011-10-01',
                'backup/2011-11-30',
                'backup/2011-12-05',
                'backup/2011-12-15',
                'backup/2011-12-30',
                ],
                [i for i in self.trimmer.walk('backup')])

        self.etc.trim_daily_limit.value = 56
        self.assertEqual([
                'backup/2011-10-01',
                'backup/2011-11-05',
                'backup/2011-11-30',
                'backup/2011-12-05',
                'backup/2011-12-15',
                'backup/2011-12-30',
                ],
                [i for i in self.trimmer.walk('backup')])

        self.etc.trim_daily_limit.value = 365
        self.assertEqual([
                'backup/2011-10-01',
                'backup/2011-11-05',
                'backup/2011-11-30',
                'backup/2011-12-05',
                'backup/2011-12-15',
                'backup/2011-12-30',
                ],
                [i for i in self.trimmer.walk('backup')])

    def test_trim_EmptyBackupDirectory(self):
        env.set_statvfs(1, 0)
        self.etc.soft_quota.value = 0

        self.assertEqual(self.trimmer.NO_NEED, self.trimmer.trim())

        self.touch('home/backup/foo')
        self.assertEqual(self.trimmer.NO_NEED, self.trimmer.trim())

        os.makedirs('home/backup/go/bar')
        self.assertEqual(self.trimmer.NO_NEED, self.trimmer.trim())

    def test_trim_RemoveUpToEmptyDirs(self):
        env.set_statvfs(2, 0)
        self.etc.soft_quota.value = 0
        self.touch('home/backup/go/00000000001000000000200000000031/2011-10-01/f',
                   'home/backup/go/00000000001000000000200000000032/2011-10-02/f',
                   )
        self.assertEqual(self.trimmer.SUCCESS, self.trimmer.trim())
        assert not exists('home/backup/go/00000000001000000000200000000031/2011-10-01')
        assert not exists('home/backup/go/00000000001000000000200000000032/2011-10-02')

    def test_trim_Lock(self):
        env.set_statvfs(1, 0)
        self.etc.soft_quota.value = 0
        self.touch('home/backup/go/00000000001000000000200000000031/2011-10-01/f')

        lock = ipc.flock('var/sugar-server/trim-backup.lock')

        self.assertEqual(self.trimmer.ONGOING, self.trimmer.trim())
        assert exists('home/backup/go/00000000001000000000200000000031/2011-10-01')

    def test_trim_LockTimeout(self):
        env.set_statvfs(2, 0)
        self.etc.soft_quota.value = 49
        self.touch('home/backup/go/00000000001000000000200000000031/2011-10-01/f')

        lock = ipc.flock('var/sugar-server/trim-backup.lock')
        lock.checkpoint()

        self.etc.trim_timeout.value = 1
        self.assertEqual(self.trimmer.SKIPPED, self.trimmer.trim())
        assert exists('home/backup/go/00000000001000000000200000000031/2011-10-01')

        time.sleep(self.etc.trim_timeout.value + 1)

        self.assertEqual(self.trimmer.SUCCESS, self.trimmer.trim())
        assert not exists('home/backup/go/00000000001000000000200000000031/2011-10-01')

    def test_trim_NoWorkOutOfSoftQuota(self):
        env.set_statvfs(2, 0)
        self.touch('home/backup/go/00000000001000000000200000000031/2011-10-01/f')
        self.etc.soft_quota.value = 50
        self.assertEqual(self.trimmer.NO_NEED, self.trimmer.trim())
        assert exists('home/backup/go/00000000001000000000200000000031/2011-10-01/f')

    def test_trim(self):
        env.set_statvfs(10, 0)

        self.touch('home/backup/go/00000000001000000000200000000031/2011-10-01/f',
                   'home/backup/go/00000000001000000000200000000031/2011-10-02/f',
                   'home/backup/go/00000000001000000000200000000032/2011-10-01/f',
                   'home/backup/go/00000000001000000000200000000032/2011-10-02/f',
                   'home/backup/go/00000000001000000000200000000033/2011-10-01/f',
                   'home/backup/go/00000000001000000000200000000033/2011-10-02/f',
                   'home/backup/go/00000000001000000000200000000034/2011-10-01/f',
                   'home/backup/go/00000000001000000000200000000034/2011-10-02/f',
                   'home/backup/go/00000000001000000000200000000035/2011-10-01/f',
                   'home/backup/go/00000000001000000000200000000035/2011-10-02/f',
                   )

        self.etc.soft_quota.value = 70
        self.assertEqual(100, env.disk_usage())

        self.assertEqual(self.trimmer.SUCCESS, self.trimmer.trim())

        self.assertEqual(50, env.disk_usage())
        assert not exists('home/backup/go/00000000001000000000200000000031/2011-10-01')
        assert exists('home/backup/go/00000000001000000000200000000031/2011-10-02/f')
        assert not exists('home/backup/go/00000000001000000000200000000032/2011-10-01')
        assert exists('home/backup/go/00000000001000000000200000000032/2011-10-02/f')
        assert not exists('home/backup/go/00000000001000000000200000000033/2011-10-01')
        assert exists('home/backup/go/00000000001000000000200000000033/2011-10-02/f')


if __name__ == '__main__':
    main()
