# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
import urllib2
from gettext import gettext as _

import active_document as ad
enforce = ad.util.enforce

from restful_document import env


LAYERS = ['public', 'deleted']

_logger = logging.getLogger('rd.document')


def restful_method(**kwargs):

    def decorate(func):
        func.is_restful_method = True
        func.restful_cls_kwargs = kwargs
        return func

    return decorate


class Document(ad.Document):
    """All RESTful document classes need to inherit this one."""

    @ad.active_property(prefix='IL', default=['public'], typecast=[LAYERS])
    def layers(self, value):
        return value

    @classmethod
    @restful_method(method='POST')
    def restful_post(cls):
        doc = cls.create(env.request.content)
        return {'guid': doc.guid}

    @classmethod
    @restful_method(method='GET')
    def restful_get(cls, **kwargs):
        offset = env.pop_int('offset', kwargs, None)
        limit = env.pop_int('limit', kwargs, None)
        query = env.pop_str('query', kwargs, None)
        reply = env.pop_list('reply', kwargs, None)
        order_by = env.pop_list('order_by', kwargs, None)

        # TODO until implementing layers support
        kwargs['layers'] = 'public'

        documents, total = cls.find(offset, limit, kwargs, query, reply,
                order_by)
        result = [i.all_properties(reply) for i in documents]

        return {'total': total.value, 'result': result}

    @restful_method(method='PUT')
    def restful_put(self, prop=None, url=None):
        if prop is None:
            self.update(self.guid, env.request.content)
        elif isinstance(self.metadata[prop], ad.BlobProperty):
            if url is not None:
                self.recv_blob(prop, url)
            else:
                self.set_blob(prop, env.request.content_stream,
                        env.request.content_length)
        else:
            self[prop] = env.request.content
            self.post()

    @restful_method(method='DELETE')
    def restful_delete(self, prop=None):
        enforce(prop is None, env.Forbidden,
                _('Properties cannot be deleted'))
        # TODO until implementing layers support
        self['layers'] = ['deleted']
        self.post()

    @restful_method(method='GET')
    def restful_get_document(self, prop=None):
        if prop is None:
            reply = []
            for name, prop in self.metadata.items():
                if isinstance(prop, ad.BrowsableProperty) and \
                        prop.permissions & ad.ACCESS_READ:
                    reply.append(name)
            return self.all_properties(reply)
        elif isinstance(self.metadata[prop], ad.BlobProperty):
            return self.send_blob(prop)
        else:
            return self[prop]

    @restful_method(method='GET', cmd='stat')
    def restful_stat_blob(self, prop=None):
        enforce(prop, _('Property name is not specified'))
        return self.stat_blob(prop)

    def recv_blob(self, prop, url):
        _logger.info(_('Download BLOB for "%s" from "%s"'), prop, url)
        stream = urllib2.urlopen(url)
        self.set_blob(prop, stream)

    def send_blob(self, prop):
        stat = self.stat_blob(prop)
        env.responce['Content-Type'] = self.metadata[prop].mime_type
        env.responce['Content-Length'] = stat['size'] if stat else 0
        return self.get_blob(prop)

    def all_properties(self, reply):
        result = {}
        for prop_name in (reply or ['guid']):
            result[prop_name] = self[prop_name]
        return result
