# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import cgi
import json
import types
import urllib
import logging
from urlparse import parse_qsl
from gettext import gettext as _

import active_document as ad
enforce = ad.util.enforce
util = ad.util

from restful_document import env


_logger = logging.getLogger('rd.router')


class Router(object):

    def __init__(self, node):
        self._metadata = _Metadata(node.documents)

        if 'SSH_ASKPASS' in os.environ:
            # Otherwise ssh-keygen will popup auth dialogs on registeration
            del os.environ['SSH_ASKPASS']

    def __call__(self, environ, start_response):
        _parse_request(environ)
        env.responce.reset()

        _logger.debug('Processing %s request %s: %s',
                env.request.method, env.request.url,
                env.request.content or '(no sent data)')

        result = None
        try:
            _authenticate()

            method = self._metadata.get_method()
            enforce(method is not None and \
                    method.method == env.request.method, env.BadRequest,
                    _('No way to handle the request'))
            result = method()

        except Exception, error:
            if isinstance(error, ad.Forbidden):
                util.exception()
                env.responce.status = env.Forbidden.status
                env.responce.update(env.Forbidden.headers)
            elif isinstance(error, env.HTTPStatus):
                env.responce.status = error.status
                env.responce.update(error.headers or {})
                result = error.result
            else:
                util.exception(_('Error while processing "%s" request'),
                        env.request.url)
                env.responce.status = '500 Internal Server Error'

            if result is None:
                env.responce['Content-Type'] = 'application/json'
                result = {'error': str(error), 'request': env.request.url}

        start_response(env.responce.status, env.responce.items())
        if isinstance(result, types.GeneratorType):
            for i in result:
                yield i
        else:
            if env.responce['Content-Type'] == 'application/json':
                result = json.dumps(result)
            yield result


class _Metadata(object):

    def __init__(self, classes):
        self._methods = [{}, {}, {}]

        for meth in _list_methods(classes):
            methods = self._methods[meth.scope].setdefault(meth.document, {})
            if meth.cmd:
                enforce(meth.cmd not in methods,
                        _('Method "%s" already exists in "%s"'),
                        meth.cmd, methods.get(meth.cmd))
                methods[meth.cmd] = meth
            else:
                enforce(meth.method not in methods,
                        _('"%s" method already exists in "%s"'),
                        meth.method, methods.get(meth.method))
                methods[meth.method] = meth

    def get_method(self):
        enforce(len(env.request.path) <= 3, env.BadRequest,
                _('Requested path consists of more than three parts'))
        if len(env.request.path) == 3:
            env.request.query['prop'] = env.request.path.pop()

        scope = len(env.request.path)
        if scope == 0:
            document = None
        else:
            document = env.request.path[0]
            enforce(document in self._methods[scope], env.BadRequest,
                    _('Unknown document type, "%s"'), document)

        if 'cmd' in env.request.query:
            method_name = env.request.query.pop('cmd')
        else:
            method_name = env.request.method

        enforce(document in self._methods[scope], env.NotFound,
                _('Not supported path'))
        methods = self._methods[scope][document]
        enforce(method_name in methods, env.NotFound,
                _('Not supported method'))

        return methods[method_name]


class _Method(object):

    def __init__(self, cls, scope, cb,
            method, cmd=None, mime_type='application/json'):
        self.cls = cls
        self.document = cls.__name__.lower() if scope else None
        self.scope = scope
        self.method = method
        self.cmd = cmd
        self.mime_type = mime_type
        self._cb = cb

    def __str__(self):
        return str(self._cb)

    def __call__(self):
        try:
            result = self._call()
        except TypeError:
            util.exception()
            raise env.BadRequest(_('Inappropriate arguments'))
        env.responce.setdefault('Content-Type', self.mime_type)
        return result

    def _call(self):
        return self._cb(**env.request.query)


class _ObjectMethod(_Method):

    def _call(self):
        guid = env.request.path[1]
        doc = self.cls(guid)
        return self._cb(doc, **env.request.query)


def _list_methods(classes):
    for cls in classes:
        for attr in [getattr(cls, i) for i in dir(cls)]:
            if not hasattr(attr, 'is_restful_method'):
                continue
            method_cls = _Method
            if isinstance(attr, types.FunctionType):
                slot = 0
            elif isinstance(attr, types.MethodType):
                if attr.im_self is not None:
                    slot = 1
                else:
                    method_cls = _ObjectMethod
                    slot = 2
            else:
                raise RuntimeError(_('Incorrect RESTful method for %r') % attr)
            yield method_cls(cls, slot, attr, **attr.restful_cls_kwargs)


def _authenticate():
    enforce('sugar_user' in env.request and \
            'sugar_user_signature' in env.request, env.Unauthorized,
            _('Sugar user credentials were not specified'))

    user = env.request['sugar_user']
    signature = env.request['sugar_user_signature']

    if env.auth.value and signature not in env.principal.authenticated and \
            (env.request.path != ['user'] or env.request.method != 'POST'):
        from restful_document.user import User
        User.verify(user, signature)
        env.principal.authenticated.add(signature)

    env.principal.user = user


def _parse_request(environ):
    request = env.request

    request.environ = environ
    path = environ['PATH_INFO'] or '/'
    __, path = urllib.splittype(path)
    __, path = urllib.splithost(path)
    request.url = path
    request.path = [i for i in path.strip('/').split('/') if i]
    request.method = environ['REQUEST_METHOD']
    request.query = {}
    request.content = None
    request.content_stream = environ.get('wsgi.input')
    request.content_length = 0

    for attr, value in parse_qsl(environ.get('QUERY_STRING', '')):
        request.query[str(attr)] = value

    if request.query:
        request.url += '?' + environ.get('QUERY_STRING')

    content_length = environ.get('CONTENT_LENGTH')
    if content_length:
        request.content_length = int(content_length)
        ctype, __ = cgi.parse_header(environ.get('CONTENT_TYPE', ''))
        if ctype.lower() == 'application/json':
            content = request.read()
            if content:
                request.content = json.loads(content)
        elif ctype.lower() == 'multipart/form-data':
            files = cgi.FieldStorage(fp=environ['wsgi.input'],
                    environ=environ)
            enforce(len(files.list) == 1,
                    _('Multipart request should contain only one file'))
            request.content_stream = files.list[0].file
