# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import hashlib
import logging
from gettext import gettext as _

from M2Crypto import DSA

import active_document as ad
enforce = ad.util.enforce
util = ad.util

from restful_document import env
from restful_document.document import Document, restful_method


_logger = logging.getLogger('rd.user')


class User(Document):

    _final = None

    @classmethod
    def init(cls, index_class):
        # Since `verify()` might be called with `restful_method.User` class
        # instance, it should know about real final document class
        # that will be sotred in `_final`.
        User._final = cls
        delattr(User, 'init')
        cls.init(index_class)

    @ad.active_property(slot=100, prefix='N', full_text=True)
    def nickname(self, value):
        return value

    @ad.active_property(ad.StoredProperty)
    def color(self, value):
        return value

    @ad.active_property(slot=101, prefix='S', permissions=ad.ACCESS_CREATE)
    def machine_sn(self, value):
        return value

    @ad.active_property(slot=102, prefix='U', permissions=ad.ACCESS_CREATE)
    def machine_uuid(self, value):
        return value

    @ad.active_property(ad.StoredProperty, permissions=ad.ACCESS_CREATE)
    def pubkey(self, value):
        return value

    @classmethod
    @restful_method(method='POST')
    def restful_post(cls):
        props = env.request.content
        enforce('pubkey' in props,
                _('Property "pubkey" is required for user registeration'))
        guid, props['pubkey'] = _load_pubkey(props['pubkey'].strip())
        doc = cls._final(**props)
        doc.set('guid', guid, raw=True)
        doc.post()
        return {'guid': guid}

    @classmethod
    def verify(cls, guid, signature):
        try:
            pubkey = cls._final(guid).get('pubkey', raw=True)
        except ad.NotFound:
            raise env.Unauthorized(_('Principal user does not exist'))
        if env.trust_users.value:
            return
        pubkey_path = util.TempFilePath(text=pubkey)
        pubkey = DSA.load_pub_key(pubkey_path)
        data = hashlib.sha1(guid).digest()
        enforce(pubkey.verify_asn1(data, signature.decode('hex')),
                env.Forbidden, _('Wrong principal credentials'))


def _load_pubkey(pubkey):
    try:
        src_path = util.TempFilePath(text=pubkey)
        # SSH key needs to be converted to PKCS8 to ket M2Crypto read it
        pubkey_pkcs8 = util.assert_call(
                ['ssh-keygen', '-f', src_path, '-e', '-m', 'PKCS8'])
    except Exception:
        message = _('Cannot read DSS public key gotten for registeration')
        util.exception(message)
        if env.trust_users.value:
            _logger.warning(_('Failed to read registration pubkey, ' \
                    'but we trust users'))
            # Keep SSH key for further converting to PKCS8
            pubkey_pkcs8 = pubkey
        else:
            raise env.Forbidden(message)

    return str(hashlib.sha1(pubkey.split()[1]).hexdigest()), pubkey_pkcs8
